const { CaptchaSolverBase } = require('./base_solver');
const { sleep, rand, waitForElement, randomMouseWiggle } = require('./helpers');

const OLD_CONTAINER = '.captcha_verify_container';
const NEW_CONTAINER = '.TUXModal.captcha-verify-container';
const FALLBACK_CONTAINERS = [
  '#captcha_container', '#tiktok-verify-ele', '.captcha-disable-scroll',
  '[class*="captcha_verify_container"]', '[class*="captcha-verify-container"]',
  '[class*="tiktok-verify-ele"]', '[class*="secsdk-captcha"]',
].join(', ');

const DETECT_TYPE_JS = `() => {
  const sels = [
    '.TUXModal.captcha-verify-container',
    '.captcha_verify_container',
    '#captcha_container',
    '#tiktok-verify-ele',
    '[class*="captcha_verify_container"]',
    '[class*="captcha-verify-container"]',
    '[class*="tiktok-verify-ele"]',
    '[class*="secsdk-captcha"]',
    '.captcha-disable-scroll'
  ];
  let c = null;
  for (const s of sels) { c = document.querySelector(s); if (c) break; }
  if (!c) return {type:"unknown",isOld:false};
  const isOld = !!document.querySelector('.captcha_verify_container');

  function isVis(el){if(!el)return false;const s=window.getComputedStyle(el);return s.display!=='none'&&s.visibility!=='hidden';}

  const allImgs = Array.from(c.querySelectorAll('img'));
  const numImgs = allImgs.length;
  const hasSlideButton = !!c.querySelector('#captcha_slide_button');
  const hasSecsdkDrag = !!c.querySelector('.secsdk-captcha-drag-icon');
  const hasCapRoundedFull = !!c.querySelector('.cap-rounded-full > div');
  const hasSlideImgClass = !!c.querySelector('.captcha_verify_img_slide') || !!c.querySelector('img.captcha_verify_img_slide');
  const capJC = c.querySelector('.cap-justify-center');

  let dragDivHasImg = false;
  if (capJC) {
    const dragDiv = capJC.querySelector('div[draggable="true"]');
    if (dragDiv) {
      const childImg = dragDiv.querySelector('img');
      if (childImg && isVis(childImg)) dragDivHasImg = true;
    }
  }

  if (hasSlideImgClass && dragDivHasImg) return {type:"pazl",isOld};
  if (dragDivHasImg) return {type:"pazl",isOld};

  const wo = c.querySelector('[data-testid="whirl-outer-img"]');
  const wi = c.querySelector('[data-testid="whirl-inner-img"]');
  if (wo && wi) return {type:"koleso",isOld};
  if (numImgs >= 2 && allImgs[1].getAttribute('draggable') === 'false') return {type:"koleso",isOld};
  const circ = allImgs.some(i=>(i.getAttribute('style')||'').includes('clip-path'));
  const abs = allImgs.some(i=>i.classList&&(i.classList.contains('cap-absolute')||i.classList.contains('absolute')));
  const hasOverlayImgs = allImgs.some(i=>{const s=window.getComputedStyle(i);return s.position==='absolute';});
  if (numImgs >= 2 && (circ || abs || hasOverlayImgs)) return {type:"koleso",isOld};
  if (numImgs >= 2 && (hasSlideButton || hasCapRoundedFull || hasSecsdkDrag)) return {type:"koleso",isOld};

  if (capJC) {
    const jcImgs = capJC.querySelectorAll('img');
    const hasCircle = Array.from(jcImgs).some(i=>(i.getAttribute('style')||'').includes('clip-path: circle'));
    if (jcImgs.length === 1 && !hasCircle && isVis(jcImgs[0])) return {type:"abc",isOld};
  }
  const verifyAction = c.querySelector('[class*="captcha_verify_action"]');
  if (verifyAction && verifyAction.querySelector('[class*="verify-captcha-submit"]')) return {type:"abc",isOld};

  if (hasSlideImgClass || hasSlideButton || hasSecsdkDrag || hasCapRoundedFull) return {type:"slider",isOld};

  if (numImgs === 1 && isVis(allImgs[0])) return {type:"abc",isOld};

  return {type:"unknown",isOld};
}`;

const GET_IMAGES_JS = `(captchaType) => {
  let c = document.querySelector('.TUXModal.captcha-verify-container')
    || document.querySelector('.captcha_verify_container')
    || document.querySelector('#captcha_container')
    || document.querySelector('[class*="captcha_verify_container"]');
  if (!c) return null;
  function b64(img){
    try{
      if(img.src&&img.src.startsWith('data:')){const m=img.src.match(/^data:[^,]+;base64,([\\s\\S]+)$/);if(m)return m[1];}
      const cv=document.createElement('canvas');cv.width=img.naturalWidth||img.width;cv.height=img.naturalHeight||img.height;
      cv.getContext('2d').drawImage(img,0,0);return cv.toDataURL('image/png').replace(/^data:image\\/\\w+;base64,/,'');
    }catch(e){
      if(img.src&&img.src.startsWith('blob:'))return null;
      if(img.src&&img.src.startsWith('data:')){const m=img.src.match(/^data:[^,]+;base64,([\\s\\S]+)$/);return m?m[1]:null;}
      return null;
    }
  }
  let images=[],mainImg=null;
  if(captchaType==='koleso'){
    const o=c.querySelector('[data-testid="whirl-outer-img"]'),i2=c.querySelector('[data-testid="whirl-inner-img"]');
    if(o&&i2){images=[o,i2];}else{
      const a=Array.from(c.querySelectorAll('img'));
      const r=a.filter(i=>!i.classList.contains('cap-absolute')),ab=a.filter(i=>i.classList.contains('cap-absolute'));
      if(r.length>0)images.push(r[0]);if(ab.length>0)images.push(ab[0]);
      if(images.length<2&&a.length>=2)images=[a[0],a[1]];
    }
  }else if(captchaType==='pazl'){
    const m=c.querySelector('#captcha-verify-image')||c.querySelector('img.captcha_verify_img_slide')||c.querySelector('.cap-justify-center img')||c.querySelector('img');
    const p=c.querySelector('div[draggable="true"] img');
    if(m&&p)images=[m,p];else{const a=Array.from(c.querySelectorAll('img'));images=a.slice(0,2);}
    mainImg=images[0]||null;
  }else if(captchaType==='slider'){
    const m=c.querySelector('#captcha-verify-image')||c.querySelector('.captcha_verify_img_slide')||c.querySelector('img');
    if(m)images=[m];mainImg=m||null;
  }else{images=Array.from(c.querySelectorAll('img'));mainImg=images[0]||null;}
  const res={images:[],nW:0,nH:0,oW:0,oH:0};
  for(const img of images){const d=b64(img);if(d)res.images.push(d);}
  if(mainImg){res.nW=mainImg.naturalWidth||0;res.nH=mainImg.naturalHeight||0;res.oW=mainImg.offsetWidth||0;res.oH=mainImg.offsetHeight||0;}
  return res;
}`;

const TASK_TEXT_JS = `() => {
  for(const s of ["span.cap-flex.cap-items-center","div[class*='captcha_verify_bar--title']","[class*='captcha_verify_bar'] span","[class*='captcha-verify'] .text"]){
    const el=document.querySelector(s);if(el&&el.textContent.trim())return el.textContent.trim();}
  return '';
}`;

const DRAG_INFO_JS = `() => {
  let c = document.querySelector('.TUXModal.captcha-verify-container')
    || document.querySelector('.captcha_verify_container')
    || document.querySelector('#captcha_container');
  if(!c) return null;
  const el=c.querySelector('.cap-rounded-full > div')||c.querySelector('.secsdk-captcha-drag-icon')||c.querySelector('#captcha_slide_button')||c.querySelector('[draggable="true"]');
  if(!el)return null;
  const r=el.getBoundingClientRect(),p=el.parentElement?el.parentElement.getBoundingClientRect():r;
  return {x:r.left,y:r.top,w:r.width,h:r.height,pW:p.width,pH:p.height};
}`;

const CONFIRM_JS = `() => {
  let c=document.querySelector('.TUXModal.captcha-verify-container')||document.querySelector('.captcha_verify_container');
  if(!c)return false;
  const b=c.querySelector('.TUXButton-label')||c.querySelector('.verify-captcha-submit-button')||c.querySelector('button[type="submit"]');
  if(b){b.click();return true;}return false;
}`;

const SUCCESS_JS = `() => {
  const c1=document.querySelector('.TUXModal.captcha-verify-container'),c2=document.querySelector('.captcha_verify_container');
  if(!c1&&!c2)return true;
  function v(e){if(!e)return false;const s=window.getComputedStyle(e);return s.display!=='none'&&s.visibility!=='hidden'&&parseFloat(s.opacity)>0.1;}
  if(!v(c1)&&!v(c2))return true;return false;
}`;

class TiktokSolver extends CaptchaSolverBase {
  async solve(selector = '') {
    this._log('Solving TikTok Captcha...');
    await randomMouseWiggle(this.page);
    for (let attempt = 0; attempt < this.attempts; attempt++) {
      this._log(`Attempt ${attempt + 1}/${this.attempts}`);

      await this._waitPageStable();

      try {
        const preSolved = await this.page.evaluate(`(() => {
          const c1 = document.querySelector('.TUXModal.captcha-verify-container');
          const c2 = document.querySelector('.captcha_verify_container');
          if (!c1 && !c2) return false;
          function v(e){if(!e)return false;const s=window.getComputedStyle(e);return s.display!=='none'&&s.visibility!=='hidden'&&parseFloat(s.opacity)>0.1;}
          return !v(c1) && !v(c2);
        })()`);
        if (preSolved) { this._log('TikTok Captcha already solved!'); return true; }
      } catch {}
      try {
        if (await this._attempt(selector || this.selector)) {
          this._log('TikTok Captcha solved!');
          return true;
        }
      } catch (e) {
        const msg = String(e).toLowerCase();
        if (msg.includes('context was destroyed') || msg.includes('navigation') || msg.includes('detached')) {
          this._log('Navigation detected, waiting...');
          await sleep(rand(3000, 6000));
          try {
            const solved = await this.page.evaluate(`(${SUCCESS_JS})()`);
            if (solved) { this._log('TikTok Captcha solved after navigation!'); return true; }
          } catch {}
        } else {
          this._log('Error:', e);
        }
        if (attempt < this.attempts - 1) await sleep(rand(2000, 4000));
      }
    }
    throw new Error('TikTok Captcha: failed. Error: ERROR_CAPTCHA_UNSOLVABLE');
  }

  async _waitPageStable() {
    for (let i = 0; i < 15; i++) {
      try {
        await this.page.evaluate(() => document.readyState);
        await sleep(500);
        await this.page.evaluate(() => document.readyState);
        return;
      } catch {
        await sleep(1000);
      }
    }
  }

  async _attempt(sel) {
    const page = this.page;

    const containerSels = sel || `${NEW_CONTAINER}, ${OLD_CONTAINER}, ${FALLBACK_CONTAINERS}`;
    const container = await waitForElement(page, containerSels, 20000);
    if (!container) throw new Error('TikTok: captcha container not found');

    await sleep(rand(500, 1500));

    let detect;
    try {
      detect = await page.evaluate(`(${DETECT_TYPE_JS})()`);
    } catch(detectErr) {
      detect = {};
    }
    if (!detect) detect = {};
    const capType = detect.type || 'unknown';
    const isOld = detect.isOld || false;
    this._log(`TikTok type=${capType}, isOld=${isOld}`);

    if (capType === 'unknown') throw new Error('TikTok: unknown captcha type');

    const rawTask = await page.evaluate(`(${TASK_TEXT_JS})()`) || '';
    this._log(`TikTok task text: ${rawTask}`);

    let textInstr;
    if (capType === 'koleso') textInstr = 'koleso';
    else if (capType === 'abc' || capType === 'objects') {
      const cleaned = rawTask ? rawTask.replace(/[:?\[\]\\-]/g, '') : '';
      textInstr = cleaned ? `abc,${cleaned}` : 'abc';
    } else textInstr = capType;

    const imgData = await page.evaluate(`(${GET_IMAGES_JS})("${capType}")`);
    if (!imgData || !imgData.images || !imgData.images.length)
      throw new Error('TikTok: no captcha images found');

    const images = imgData.images;
    const nW = imgData.nW || 0, nH = imgData.nH || 0;
    const oW = imgData.oW || 0, oH = imgData.oH || 0;
    this._log(`TikTok: ${images.length} images, natural=${nW}x${nH}, offset=${oW}x${oH}`);

    let rawResult;
    if (capType === 'koleso') {
      if (images.length < 2) throw new Error('TikTok koleso needs 2 images');
      rawResult = await this.client.solveCoordinatesRaw({
        click: 'geetest', textinstructions: 'koleso', body0: images[0], body1: images[1], debug: this.debug,
      });
    } else {
      rawResult = await this.client.solveCoordinatesRaw({
        click: 'geetest', textinstructions: textInstr, body: images[0], debug: this.debug,
      });
    }

    if (!rawResult) throw new Error('TikTok: empty API response');
    this._log(`TikTok result: ${rawResult}`);

    if (capType === 'koleso') return await this._handleKoleso(page, rawResult);
    if (capType === 'slider') return await this._handleSlider(page, rawResult, nH, oH, isOld);
    if (capType === 'pazl') return await this._handlePazl(page, rawResult, nW, oW);
    return await this._handleClick(page, rawResult, nW, nH, oW, oH);
  }

  async _handleKoleso(page, raw) {
    const parts = raw.replace(/[^0-9,]/g, '').split(',');
    const angle = parts.length > 1 ? parseInt(parts[1]) : parseInt(parts[0]);
    this._log(`koleso angle=${angle}`);

    const di = await page.evaluate(`(${DRAG_INFO_JS})()`);
    if (!di) throw new Error('koleso: drag element not found');

    const maxMove = di.pW - di.w;
    const offset = Math.round((angle / 271) * maxMove);
    const sx = di.x + di.w / 2;
    const sy = di.y + di.h / 2;

    this._log(`koleso: offset=${offset}, parentW=${di.pW}, elW=${di.w}`);
    await this._drag(page, sx, sy, sx + offset, sy);
    await sleep(rand(2000, 3000));
    return await this._checkSuccess(page);
  }

  async _handleSlider(page, raw, nH, oH, isOld) {
    const parts = raw.replace(/[^0-9,]/g, '').split(',');
    let res1 = parseInt(parts[0]);
    this._log(`slider raw=${res1}, nH=${nH}, oH=${oH}`);

    if (nH > 0 && oH > 0) { res1 = Math.round(res1 / nH * oH); this._log(`slider scaled=${res1}`); }
    const dist = res1 - 5;

    if (isOld) {
      const el = await page.$('.secsdk-captcha-drag-icon');
      if (el) {
        const box = await el.boundingBox();
        if (box) await this._drag(page, box.x + box.width / 2, box.y + box.height / 2, box.x + box.width / 2 + dist, box.y + box.height / 2);
      }
    } else {
      const di = await page.evaluate(`(${DRAG_INFO_JS})()`);
      if (di) {
        const sx = di.x + di.w / 2, sy = di.y + di.h / 2;
        await this._drag(page, sx, sy, sx + dist, sy);
      } else throw new Error('slider: drag element not found');
    }

    await sleep(1000);
    if (!isOld) await page.evaluate(`(${CONFIRM_JS})()`);
    await sleep(rand(2000, 3000));
    return await this._checkSuccess(page);
  }

  async _handlePazl(page, raw, nW, oW) {
    const parts = raw.replace(/[^0-9,]/g, '').split(',');
    let res1 = parseInt(parts[0]);
    this._log(`pazl raw=${res1}, nW=${nW}, oW=${oW}`);

    if (nW > 0 && oW > 0) { res1 = Math.round(res1 / nW * oW); this._log(`pazl scaled=${res1}`); }
    const dist = res1 - 5;

    const di = await page.evaluate(`(${DRAG_INFO_JS})()`);
    if (!di) throw new Error('pazl: drag element not found');

    const sx = di.x + di.w / 2, sy = di.y + di.h / 2;
    await this._drag(page, sx, sy, sx + dist, sy);
    await sleep(1000);
    await page.evaluate(`(${CONFIRM_JS})()`);
    await sleep(rand(2000, 3000));
    return await this._checkSuccess(page);
  }

  async _handleClick(page, raw, nW, nH, oW, oH) {
    const clean = raw.replace(/[^0-9,;]/g, '');
    const groups = clean.split(';');

    const img = await page.$(
      '#captcha-verify-image, .TUXModal.captcha-verify-container img, ' +
      '.captcha_verify_container img, div[class*="captcha"][class*="verify"] img'
    );
    if (!img) throw new Error('click: image not found');
    const box = await img.boundingBox();
    if (!box) throw new Error('click: image box not found');

    for (const g of groups) {
      const p = g.split(',');
      if (p.length < 2) continue;
      let x = parseInt(p[0]), y = parseInt(p[1]);
      if (nW > 0 && oW > 0) x = Math.round(x / nW * oW);
      if (nH > 0 && oH > 0) y = Math.round(y / nH * oH);
      this._log(`click: (${x},${y})`);
      await sleep(500);
      await page.mouse.click(box.x + x, box.y + y);
      await sleep(1200);
    }

    await sleep(2000);
    const submit = await page.$('.TUXButton-label, .verify-captcha-submit-button, button[type="submit"]');
    if (submit) await submit.click();
    await sleep(rand(2000, 3000));
    return await this._checkSuccess(page);
  }

  async _drag(page, sx, sy, ex, ey, steps = 20) {
    await page.mouse.move(sx, sy);
    await page.mouse.down();
    await sleep(100);
    const dx = (ex - sx) / steps;
    for (let i = 1; i <= steps; i++) {
      await page.mouse.move(sx + dx * i + rand(-2, 2), sy + rand(-1, 1));
      await sleep(rand(30, 70));
    }
    await page.mouse.move(ex, ey);
    await sleep(100);
    await page.mouse.up();
    await sleep(300);
  }

  async _checkSuccess(page) {
    await sleep(1000);
    return await page.evaluate(`(${SUCCESS_JS})()`);
  }
}

module.exports = { TiktokSolver };
