const { CaptchaSolverBase } = require('./base_solver');
const { sleep, rand, waitForElement, randomMouseWiggle } = require('./helpers');

class RecaptchaSolver extends CaptchaSolverBase {
  async solve() {
    this._log('Solving reCAPTCHA v2...');
    await randomMouseWiggle(this.page);
    for (let attempt = 0; attempt < this.attempts; attempt++) {
      this._log(`Attempt ${attempt + 1}/${this.attempts}`);
      try {
        if (await this._attempt()) {
          this._log('reCAPTCHA v2 solved!');
          return true;
        }
      } catch (e) {
        this._log('Error in attempt:', e);
        if (attempt < this.attempts - 1) await sleep(rand(1000, 2000));
      }
    }
    throw new Error('reCAPTCHA v2: failed after all attempts. Error: ERROR_CAPTCHA_UNSOLVABLE');
  }

  async _attempt() {
    const page = this.page;

    const checkboxFrameEl = await waitForElement(page, '>CSS> iframe[src*="recaptcha"][src*="anchor"]', 10000);
    if (!checkboxFrameEl) throw new Error('reCAPTCHA iframe not found');
    const checkboxFrame = await checkboxFrameEl.contentFrame();
    const checkbox = await checkboxFrame.$('#recaptcha-anchor');
    if (!checkbox) throw new Error('reCAPTCHA checkbox not found');

    if (!(await checkboxFrame.$('#recaptcha-anchor[aria-checked="true"]'))) {
      this._log('Clicking reCAPTCHA checkbox...');
      await sleep(rand(200, 600));
      await checkboxFrame.evaluate(el => el.click(), checkbox);
      await sleep(rand(1500, 3000));
    }
    if (await checkboxFrame.$('#recaptcha-anchor[aria-checked="true"]')) {
      this._log('reCAPTCHA solved without image challenge!');
      return true;
    }

    const challengeFrameEl = await waitForElement(page, '>CSS> iframe[src*="recaptcha"][src*="bframe"]', 10000);
    if (!challengeFrameEl) throw new Error('reCAPTCHA challenge iframe not found');
    const cf = await challengeFrameEl.contentFrame();

    await sleep(1000);
    const payloadEl = await waitForElement(cf, '>CSS> .rc-imageselect-payload', 8000);
    if (!payloadEl) throw new Error('reCAPTCHA payload not found');
    await this._onImagesReady(cf);
    await sleep(1000);

    const hasTiles = await cf.$('.rc-image-tile-44, .rc-image-tile-33');
    if (!hasTiles) throw new Error('reCAPTCHA: tile elements not found');

    let repeatClick = false;
    let hashUrls = null;
    let repeatError = 0;
    let repeatSubmit = 0;

    for (let i = 0; i < 25; i++) {
      if (repeatSubmit > 3) {
        this._log('Solved (many submits)');
        return true;
      }

      const selected = await cf.$$('.rc-imageselect-tileselected');
      if (selected.length > 0) {
        repeatSubmit++;
        await this._submit(cf);
        continue;
      }

      await this._waitDynamic(cf);
      repeatSubmit = 0;

      if (await this._isSolved(cf, checkboxFrame)) return true;

      const taskText = await this._getTaskText(cf);
      if (!taskText) { await sleep(1000); continue; }

      await this._waitStable(cf);

      const tileData = await cf.evaluate(() => {
        const imgs = document.querySelectorAll('.rc-imageselect-target img');
        const result = [];
        let idx = 0;
        for (const img of imgs) {
          if (!img.complete || img.naturalWidth === 0) return null;
          let url = null;
          const bg = img.parentElement?.style?.background?.trim();
          if (bg) {
            const m = bg.match(/(?!^)".*?"/g);
            if (m && m.length > 0) url = m[0].replace(/"/g, '');
          }
          if (!url) url = img.src;
          result.push({ w: img.naturalWidth, url, idx });
          idx++;
        }
        return result;
      });

      if (!tileData || tileData.length === 0) {
        this._log('Images not loaded, retrying');
        await sleep(1000);
        continue;
      }

      const unique = [];
      const seen = new Set();
      for (const t of tileData) {
        if (!seen.has(t.url)) {
          seen.add(t.url);
          unique.push(t);
        }
      }

      const currentHash = JSON.stringify(unique.map(u => u.url));
      if (!repeatClick && hashUrls !== null && hashUrls === currentHash) {
        repeatError++;
        if (repeatError > 3) {
          this._log('Images repeating, reloading');
          repeatError = 0;
          await this._reload(cf);
          await sleep(3000);
          continue;
        }
        this._log('Images repeat, waiting');
        await sleep(3000);
        continue;
      }

      const tile11 = await cf.$('.rc-imageselect-table-33 .rc-image-tile-11');
      repeatClick = !!tile11;
      hashUrls = currentHash;

      let setka = 3;
      if (unique.length > 1) {
        const filtered = [];
        for (const u of unique) {
          if (u.w > 150) {
            setka = 1;
          } else {
            filtered.push(u);
          }
        }
        if (setka === 1 && filtered.length > 0) {
          unique.length = 0;
          unique.push(...filtered);
        }
      } else {
        for (const u of unique) {
          if (u.w > 350) setka = 4;
        }
      }

      const imgList = [];
      let imgError = false;
      for (const u of unique) {
        const b64 = await cf.evaluate((url) => {
          return new Promise((resolve) => {
            const img = new Image();
            img.crossOrigin = 'anonymous';
            img.onload = function() {
              const canvas = document.createElement('canvas');
              canvas.width = this.width;
              canvas.height = this.height;
              canvas.getContext('2d').drawImage(this, 0, 0);
              resolve(canvas.toDataURL('image/png').replace(/^data:image\/(png|jpg);base64,/, ''));
            };
            img.onerror = () => resolve(null);
            img.src = url;
          });
        }, u.url);
        if (b64) {
          imgList.push(b64);
        } else {
          this._log('Failed to get image:', u.url.slice(0, 60));
          imgError = true;
          break;
        }
      }

      if (imgError) { await sleep(1000); continue; }

      this._log(`Sending ${imgList.length} image(s), setka=${setka}, task="${taskText.slice(0, 50)}"`);

      const payload = {
        type: 'base64',
        click: 'recap2',
        textinstructions: taskText,
        sizex: setka,
      };

      if (imgList.length === 1) {
        payload.body = imgList[0];
      } else {
        for (let k = 0; k < imgList.length; k++) {
          payload['body' + k] = imgList[k];
        }
      }

      const result = await this.client.click(payload, this.debug);

      if (!result || result.includes('ERROR') || result.includes('WAIT')) {
        this._log('API error, reloading:', result);
        await this._reload(cf);
        await sleep(3000);
        continue;
      }

      if (result === 'notpic') {
        this._log('notpic — submitting');
        await this._submit(cf);
        continue;
      }

      const indices = result.replace(/[^0-9,]/g, '').split(',').map(Number).filter(n => !isNaN(n) && n > 0);
      if (!indices.length) {
        this._log('No valid indices in result:', result);
        await this._reload(cf);
        continue;
      }

      this._log('Clicking tiles:', indices);

      for (const num of indices) {
        await sleep(rand(300, 600));
        let clickIdx = num - 1;
        if (setka === 1) {
          const orig = unique[clickIdx];
          if (orig) clickIdx = orig.idx;
        }

        await cf.evaluate((idx) => {
          const selector = document.querySelector('.rc-imageselect-table-33, .rc-imageselect-table-42, .rc-imageselect-table-44');
          if (!selector) return;
          const tile = selector.querySelectorAll('.rc-imageselect-tile')[idx];
          if (!tile) return;
          const rect = tile.getBoundingClientRect();
          const cx = rect.left + rect.width / 2;
          const cy = rect.top + rect.height / 2;
          ['mouseover', 'mousedown', 'mouseup', 'click'].forEach(type => {
            tile.dispatchEvent(new MouseEvent(type, {
              bubbles: true, cancelable: true, view: window, clientX: cx, clientY: cy
            }));
          });
        }, clickIdx);
      }

      const dynAfterClick = await cf.$('.rc-imageselect-table-33 .rc-image-tile-11, .rc-imageselect-dynamic-selected');
      if (dynAfterClick) repeatClick = true;

      await sleep(1000);

      if (repeatClick) {
        this._log('Repeat click mode, continuing');
        await sleep(2000);
        continue;
      }

      await this._submit(cf);
      await sleep(3000);

      if (await this._isSolved(cf, checkboxFrame)) return true;
    }

    throw new Error('reCAPTCHA v2: could not solve after 25 iterations');
  }

  async _getTaskText(cf) {
    return await cf.evaluate(() => {
      let task = document.querySelector('.rc-imageselect-instructions')?.innerText?.replace(/\s+/g, ' ')?.trim();
      if (!task) return '';
      task = task.replace('Click verify once there are none left', '');
      task = task.replace('If there are none, click skip', '');
      return task.trim();
    });
  }

  async _waitDynamic(cf, maxWait = 60) {
    for (let i = 0; i < maxWait; i++) {
      const dyn = await cf.$$('.rc-imageselect-dynamic-selected');
      if (!dyn.length) return;
      await sleep(1000);
    }
  }

  async _onImagesReady(cf, timeout = 20000) {
    const start = Date.now();
    while (Date.now() - start < timeout) {
      const ready = await cf.evaluate(() => {
        const payload = document.querySelector('.rc-imageselect-payload');
        if (!payload) return false;
        const tiles = document.querySelectorAll('.rc-imageselect-tile');
        const loading = document.querySelectorAll('.rc-imageselect-dynamic-selected');
        if (tiles.length === 0 || loading.length > 0) return false;
        const imgs = document.querySelectorAll('.rc-imageselect-target img');
        for (const img of imgs) {
          if (!img.complete || img.naturalWidth === 0) return false;
        }
        return true;
      });
      if (ready) {
        await sleep(500);
        const still = await cf.evaluate(() => {
          const tiles = document.querySelectorAll('.rc-imageselect-tile');
          const loading = document.querySelectorAll('.rc-imageselect-dynamic-selected');
          return tiles.length > 0 && loading.length === 0;
        });
        if (still) return true;
      }
      await sleep(200);
    }
    return false;
  }

  async _waitStable(cf) {
    for (let i = 0; i < 10; i++) {
      const before = await cf.evaluate(() => document.querySelectorAll('.rc-imageselect-target img').length);
      await sleep(300);
      const after = await cf.evaluate(() => document.querySelectorAll('.rc-imageselect-target img').length);
      const dynamic = await cf.evaluate(() => document.querySelectorAll('.rc-imageselect-dynamic-selected').length);
      if (before === after && before > 0 && dynamic === 0) return;
    }
  }

  async _isSolved(cf, checkboxFrame) {
    try {
      const cbSolved = await checkboxFrame.$('.recaptcha-checkbox[aria-checked="true"]');
      if (cbSolved) return true;
    } catch {}
    return await cf.evaluate(() => {
      const submit = document.querySelector('#recaptcha-verify-button');
      const reload = document.querySelector('.rc-button-reload');
      if (submit && submit.disabled && reload && (' ' + reload.className + ' ').indexOf(' rc-button-disabled ') > -1) return true;
      if (document.querySelectorAll('.rc-imageselect-correct, .rc-imageselect-success').length > 0) return true;
      return false;
    });
  }

  async _submit(cf) {
    await cf.evaluate(() => {
      const btn = document.querySelector('#recaptcha-verify-button');
      if (!btn) return;
      ['mousedown', 'mouseup', 'click'].forEach(type => {
        btn.dispatchEvent(new MouseEvent(type, { bubbles: true, cancelable: true, view: window }));
      });
    });
    await sleep(rand(2500, 3500));
  }

  async _reload(cf) {
    await cf.evaluate(() => {
      const el = document.querySelector('.rc-button-reload');
      if (!el) return;
      ['mousedown', 'mouseup', 'click'].forEach(type => {
        el.dispatchEvent(new MouseEvent(type, { bubbles: true, cancelable: true, view: window }));
      });
    });
    await sleep(rand(1000, 1500));
  }
}

module.exports = { RecaptchaSolver };
