const { CaptchaSolverBase } = require('./base_solver');
const { sleep, rand, waitForElement, screenshotElement, canvasToBase64, humanClick, randomMouseWiggle } = require('./helpers');

class OtherSolver extends CaptchaSolverBase {
  async solve(selector = '') {
    this._log('Solving Other Captcha...');
    await randomMouseWiggle(this.page);
    const sel = selector || this.selector;
    const maxIterations = this.attempts + 3;
    let errors = 0;
    for (let i = 0; i < maxIterations; i++) {
      try {
        const result = await this._attempt(sel);
        if (result === true) { this._log('Other Captcha solved!'); return true; }
        if (result === false) { await sleep(rand(1000, 2000)); continue; }
      } catch (e) {
        this._log('Error:', e);
        errors++;
        if (errors >= this.attempts) break;
        await sleep(rand(1500, 3000));
      }
    }
    throw new Error('Other Captcha: failed.');
  }

  async _attempt(sel) {
    const page = this.page;
    let captchaFrame = page;

    if (sel) {
      const iframeEl = await waitForElement(page, sel, 10000);
      if (iframeEl) captchaFrame = (await iframeEl.contentFrame()) || page;
    } else {
      const autoEl = await waitForElement(page, '>CSS> iframe[src*="smartcaptcha"], iframe[src*="captcha-api"]', 8000);
      if (autoEl) captchaFrame = (await autoEl.contentFrame()) || page;
    }

    const checkbox = await captchaFrame.$('.CheckboxCaptcha-Anchor');
    const slider = await captchaFrame.$('.CaptchaSlider');

    if (checkbox && await this._isVisible(captchaFrame, '.CheckboxCaptcha-Anchor')) {
      const className = await checkbox.evaluate(el => el.getAttribute('class') || '');
      if (!className.includes('pending')) {
        this._log('Phase 1: clicking checkbox');
        await sleep(rand(800, 1500));
        const inp = await captchaFrame.$('.CheckboxCaptcha-Anchor input');
        if (inp) { const [cx, cy] = await this._centerOf(inp); await humanClick(page, cx, cy); }
        else { const [cx, cy] = await this._centerOf(checkbox); await humanClick(page, cx, cy); }
      } else { this._log('Phase 1: checkbox already pending'); }

      this._log('Checkbox clicked, waiting for captcha to load...');
      await sleep(rand(3000, 5000));

      const advAfterCheckbox = await this._detectAdvanced(captchaFrame).catch(() => null);
      if (advAfterCheckbox) {
        this._log(`Advanced captcha appeared after checkbox: ${advAfterCheckbox}`);
      } else {
        return false;
      }
    }

    if (slider && await this._isVisible(captchaFrame, '.CaptchaSlider')) {
      const advancedAlready = await this._detectAdvanced(captchaFrame);
      if (advancedAlready) {
        this._log(`Phase 1: slider visible but advanced captcha already present (${advancedAlready}), skipping drag`);
      } else {
        this._log('Phase 1: dragging slider');
        await sleep(rand(600, 1200));
        const thumb = await captchaFrame.$('.CaptchaSlider .Thumb');
        if (thumb) {
          const sliderBox = await slider.boundingBox();
          if (sliderBox) await this._dragElement(page, thumb, sliderBox.width);
        }
        await sleep(rand(2000, 3500));
      }
    }

    let detected;
    try {
      detected = await this._detectAdvanced(captchaFrame);
    } catch (e) {
      const msg = String(e).toLowerCase();
      if (msg.includes('context was destroyed') || msg.includes('navigation') || msg.includes('detached')) {
        this._log('Page navigated — captcha solved');
        return true;
      }
      throw e;
    }

    if (detected === 'figur') { this._log('Detected: figur'); return await this._solveTwoImagesV2(page, captchaFrame); }
    if (detected === 'kaleidoscope') { this._log('Detected: kaleidoscope'); return await this._solveKaleidoscope(page, captchaFrame); }
    if (detected === 'text') { this._log('Detected: text captcha'); return await this._solveTextInput(page, captchaFrame); }
    if (detected === 'image_click') { this._log('Detected: image click'); return await this._solveImageClick(page, captchaFrame); }
    if (detected === 'two_images') { this._log('Detected: two-image click'); return await this._solveTwoImages(page, captchaFrame); }
    if (detected === 'kaleidoscope_canvas') { this._log('Detected: kaleidoscope canvas'); return await this._solveKaleidoscope(page, captchaFrame); }

    let stillVisible = false;
    try {
      await sleep(rand(1000, 2000));
      for (const selCheck of ['.CheckboxCaptcha-Anchor', '.CaptchaSlider', '.AdvancedCaptcha-ImageWrapper', '.AdvancedCaptcha_kaleidoscope', '.AdvancedCaptcha-View>img']) {
        const el = await captchaFrame.$(selCheck);
        if (el && await this._isVisible(captchaFrame, selCheck)) { stillVisible = true; break; }
      }
    } catch (e) {
      const msg = String(e).toLowerCase();
      if (msg.includes('context was destroyed') || msg.includes('navigation') || msg.includes('detached')) {
        this._log('Page navigated — captcha solved');
        return true;
      }
      throw e;
    }

    if (!stillVisible) { this._log('No captcha elements visible — likely solved'); return true; }
    throw new Error('Other: unknown captcha type');
  }

  async _detectAdvanced(captchaFrame) {
    if (await captchaFrame.$('.AdvancedCaptcha-ImageWrapper') && await this._isVisible(captchaFrame, '.AdvancedCaptcha-ImageWrapper')) return 'figur';
    if (await captchaFrame.$('.AdvancedCaptcha_kaleidoscope') && await this._isVisible(captchaFrame, '.AdvancedCaptcha_kaleidoscope')) return 'kaleidoscope';
    if (await captchaFrame.$('.AdvancedCaptcha-View>img') && await this._isVisible(captchaFrame, '.AdvancedCaptcha-View>img')) return 'text';
    if (await captchaFrame.$('img[src*="/captchaimage"], img[src*="/captchaimg"]')) return 'image_click';
    if (await captchaFrame.$('img[src*="data=img"]') && await captchaFrame.$('img[src*="data=task"]')) return 'two_images';
    if (await captchaFrame.$('canvas.AdvancedCaptcha-KaleidoscopeCanvas')) return 'kaleidoscope_canvas';
    return null;
  }

  async _fetchImageB64(pageOrFrame, url) {
    if (!url) return '';
    try {
      const b64 = await pageOrFrame.evaluate(async (url) => {
        try {
          const r = await fetch(url, { credentials: 'include' });
          if (!r.ok) return null;
          const blob = await r.blob();
          return await new Promise((res, rej) => {
            const reader = new FileReader();
            reader.onloadend = () => res(reader.result.split(',')[1]);
            reader.onerror = rej;
            reader.readAsDataURL(blob);
          });
        } catch { return null; }
      }, url);
      if (b64) return b64;
    } catch {}

    try {
      const b64 = await pageOrFrame.evaluate((url) => {
        try {
          const imgs = document.querySelectorAll('img');
          for (const img of imgs) {
            if (img.src === url || img.src.includes(url)) {
              const c = document.createElement('canvas');
              c.width = img.naturalWidth || img.width;
              c.height = img.naturalHeight || img.height;
              c.getContext('2d').drawImage(img, 0, 0);
              return c.toDataURL('image/png').split(',')[1];
            }
          }
        } catch {}
        return null;
      }, url);
      if (b64) return b64;
    } catch {}

    try {
      const b64 = await this._serverFetchB64(url);
      if (b64) return b64;
    } catch {}

    try {
      const result = await this._urlToBase64(url);
      if (Array.isArray(result)) return result[0] || '';
      return result || '';
    } catch { return ''; }
  }

  static _buildTaskUrl(url) {
    if (!url) return '';
    if (/[?&]b?data=task\b/i.test(url)) return url;
    const m = url.match(/([?&])(b?data)=img\b/i);
    if (m) return url.slice(0, m.index) + m[1] + m[2] + '=task' + url.slice(m.index + m[0].length);
    const sep = url.includes('?') ? '&' : '?';
    return url + sep + 'data=task';
  }

  async _isVisible(frame, selector) {
    try {
      return await frame.$eval(selector, (el) => {
        if (!el) return false;
        const s = window.getComputedStyle(el);
        return !!(el.offsetWidth || el.offsetHeight || el.getClientRects().length)
          && s.visibility !== 'hidden' && s.display !== 'none';
      });
    } catch { return false; }
  }

  async _centerOf(element) {
    const box = await element.boundingBox();
    if (!box) return [0, 0];
    return [box.x + box.width / 2, box.y + box.height / 2];
  }

  async _dragElement(page, element, trackWidth) {
    try { await page.mouse.up(); } catch {}
    await sleep(rand(100, 200));

    const thumbBox = await element.boundingBox();
    if (!thumbBox) return;

    let sliderBox = null;
    try {
      sliderBox = await element.evaluate(el => {
        const p = el.closest('.CaptchaSlider');
        if (!p) return null;
        const r = p.getBoundingClientRect();
        return { x: r.left, y: r.top, width: r.width, height: r.height };
      });
    } catch {}

    const startX = thumbBox.x + thumbBox.width / 2;
    const startY = thumbBox.y + thumbBox.height / 2;

    let distance;
    if (sliderBox) {
      const trackEnd = sliderBox.x + sliderBox.width - thumbBox.width / 2;
      distance = trackEnd - startX;
    } else {
      distance = trackWidth;
    }

    if (distance <= 5) return;
    const targetX = startX + distance;

    await page.mouse.move(startX, startY);
    await sleep(rand(150, 400));
    await page.mouse.down();
    await sleep(rand(50, 150));

    const steps = Math.max(20, Math.floor(distance / 2.5) + rand(15, 35));
    let yDrift = 0;
    const yDriftDir = Math.random() < 0.5 ? 1 : -1;

    for (let i = 0; i <= steps; i++) {
      const progress = i / steps;
      let eased;
      if (progress < 0.5) eased = 4 * progress * progress * progress;
      else eased = 1 - Math.pow(-2 * progress + 2, 3) / 2;

      const currentX = startX + distance * eased;
      yDrift += (Math.random() - 0.5) * 0.8 * yDriftDir;
      yDrift *= 0.90;
      const currentY = startY + yDrift + (Math.random() - 0.5) * 2.4;

      await page.mouse.move(currentX, currentY);

      const speedCurve = Math.sin(progress * Math.PI);
      const baseWait = 16 / (0.8 + speedCurve * 0.4);
      let extraWait = 0;
      if (Math.random() < 0.04) extraWait = Math.abs((Math.random() - 0.5) * 100);
      const waitMs = Math.max(6, baseWait + (Math.random() - 0.5) * 6 + extraWait);
      await sleep(Math.floor(waitMs));
    }

    await sleep(rand(80, 250));
    await page.mouse.up();
  }

  async _solveImageClick(page, frame) {
    const mainImg = await frame.$('img[src*="/captchaimage"], img[src*="/captchaimg"]');
    const imgSrc = await mainImg.evaluate(el => el.src);
    const b64 = imgSrc.startsWith('http')
      ? await this._fetchImageB64(frame, imgSrc)
      : await screenshotElement(mainImg);

    const result = await this.client.click({ type: 'base64', vernet: '18', body: b64 }, this.debug);
    if (!result) throw new Error('Other type 1: no result');

    const imgDims = await mainImg.evaluate(el => ({
      natW: el.naturalWidth || el.width, natH: el.naturalHeight || el.height,
      cssW: el.clientWidth || el.offsetWidth, cssH: el.clientHeight || el.offsetHeight,
    }));
    const coordStr = result.includes('coordinates:') ? result.split('coordinates:')[1] : result;
    const imgBox = await mainImg.boundingBox();

    for (const pair of coordStr.split(';').map(c => c.trim()).filter(Boolean)) {
      const xVal = parseInt(pair.split(',')[0].split('=')[1].replace(/[^0-9]/g, ''));
      const yVal = parseInt(pair.split(',')[1].split('=')[1].replace(/[^0-9]/g, ''));
      await humanClick(page,
        imgBox.x + xVal / (imgDims.natW || 300) * (imgDims.cssW || 300),
        imgBox.y + yVal / (imgDims.natH || 200) * (imgDims.cssH || 200),
      );
    }

    await sleep(rand(500, 1000));
    const btn = await frame.$('button[type="submit"]');
    if (btn) await btn.click({ delay: rand(30, 80) });
    await sleep(rand(1500, 2500));
    return !(await frame.$('img[src*="/captchaimage"], img[src*="/captchaimg"]'));
  }

  async _solveTwoImages(page, frame) {
    const dataImgEl = await frame.$('img[src*="data=img"]');
    const dataTaskEl = await frame.$('img[src*="data=task"]');
    const body0 = await this._fetchImageB64(frame, await dataImgEl.evaluate(el => el.src));
    const body1 = await this._fetchImageB64(frame, await dataTaskEl.evaluate(el => el.src));

    const rawResult = await this.client.click({
      type: 'base64', body0, body1, click: 'oth', textinstructions: 'yandex',
    }, this.debug);

    const coordStr = rawResult.includes('coordinates:') ? rawResult.split('coordinates:')[1] : rawResult;
    const imgBox = await dataImgEl.boundingBox();
    const imgDims = await dataImgEl.evaluate(el => ({
      natW: el.naturalWidth || el.width, natH: el.naturalHeight || el.height,
      cssW: el.clientWidth || el.offsetWidth, cssH: el.clientHeight || el.offsetHeight,
    }));

    for (const pair of coordStr.split(';').map(c => c.trim()).filter(Boolean)) {
      const xVal = parseInt(pair.split(',')[0].split('=')[1].replace(/[^0-9]/g, ''));
      const yVal = parseInt(pair.split(',')[1].split('=')[1].replace(/[^0-9]/g, ''));
      await humanClick(page,
        imgBox.x + xVal / (imgDims.natW || 300) * (imgDims.cssW || 300),
        imgBox.y + yVal / (imgDims.natH || 200) * (imgDims.cssH || 200),
      );
    }

    await sleep(rand(500, 1000));
    const btn = await frame.$('button[type="submit"]');
    if (btn) await btn.click({ delay: rand(30, 80) });
    await sleep(rand(1500, 2500));
    return !(await frame.$('img[src*="data=img"]'));
  }

  async _solveTwoImagesV2(page, frame) {
    const imgEl = await frame.$('.AdvancedCaptcha-ImageWrapper img');
    if (!imgEl) throw new Error('Other figur: img not found');

    const imgSrc = await imgEl.evaluate(el => el.src);
    const body0 = await this._fetchImageB64(frame, imgSrc);

    const taskUrl = OtherSolver._buildTaskUrl(imgSrc);
    let body1 = '';
    if (taskUrl) { try { body1 = await this._fetchImageB64(frame, taskUrl); } catch {} }
    if (!body1) { try { body1 = await canvasToBase64(frame, '.AdvancedCaptcha-View canvas'); } catch {} }
    if (!body0 || !body1) throw new Error('Other figur: failed to extract images');

    const rawResult = await this.client.click({
      type: 'base64', body0, body1, click: 'oth', textinstructions: 'yandex',
    }, this.debug);

    let coordStr = rawResult.includes('coordinates:') ? rawResult.split('coordinates:')[1] : rawResult;
    coordStr = coordStr.replace(/[^0-9,;]/g, '');

    const wrapper = await frame.$('.AdvancedCaptcha-ImageWrapper');
    const wrapperBox = await wrapper.boundingBox();

    for (const pair of coordStr.split(';').map(c => c.trim()).filter(Boolean)) {
      const parts = pair.split(',');
      if (parts.length >= 2) {
        const xVal = parseInt(parts[0]);
        const yVal = parseInt(parts[1]);
        await sleep(rand(200, 600));
        await humanClick(page, wrapperBox.x + xVal, wrapperBox.y + yVal);
      }
    }

    await sleep(rand(1500, 2500));
    let btn = await frame.$('button[data-testid="submit"]');
    if (!btn) btn = await frame.$('button[type="submit"]');
    if (btn) await btn.click({ delay: rand(30, 80) });
    await sleep(rand(2500, 3500));
    return !(await frame.$('.AdvancedCaptcha-ImageWrapper'));
  }

  async _solveKaleidoscope(page, frame) {
    let puzzle = '';
    let imgSrc = '';

    try {
      const ssr = await frame.evaluate(() => {
        if (window.__SSR_DATA__ && window.__SSR_DATA__.imageSrc && window.__SSR_DATA__.task) {
          return {
            imageSrc: window.__SSR_DATA__.imageSrc,
            task: Array.isArray(window.__SSR_DATA__.task) ? window.__SSR_DATA__.task.join(',') : String(window.__SSR_DATA__.task),
          };
        }
        return null;
      });
      if (ssr) { imgSrc = ssr.imageSrc; puzzle = ssr.task; }
    } catch {}

    if (!puzzle || !imgSrc) {
      const bodyHtml = await frame.evaluate(() => document.body ? document.body.innerHTML : '');
      const taskMatch = bodyHtml.match(/"task"\s*:\s*\[([^\]]+)\]/) || bodyHtml.match(/\[([0-9,\s]+)\]/);
      const imgMatch = bodyHtml.match(/"imageSrc"\s*:\s*"(.*?)"/) || bodyHtml.match(/,imageSrc:"(.*?)"/);

      if (taskMatch && imgMatch) {
        puzzle = taskMatch[1].replace(/[^0-9,]/g, '');
        imgSrc = imgMatch[1];
      } else {
        const canvasSel = 'canvas.AdvancedCaptcha-KaleidoscopeCanvas';
        let canvasEl = await frame.$(canvasSel);
        if (!canvasEl) canvasEl = await frame.$('.AdvancedCaptcha-View canvas');
        if (canvasEl) {
          const b64 = await canvasToBase64(frame, canvasSel);
          let taskText = '';
          try { taskText = await frame.$eval('.AdvancedCaptcha-Task', el => el.textContent.trim()); } catch {}
          const rawResult = await this.client.click({ type: 'base64', body: b64, click: 'oth2', textinstructions: taskText }, this.debug);
          const cStr = rawResult.includes('coordinates:') ? rawResult.split('coordinates:')[1] : rawResult;
          const box = await canvasEl.boundingBox();
          const canvasDims = await frame.$eval(canvasSel, (c) => ({ w: c.width, h: c.height, cssW: c.clientWidth, cssH: c.clientHeight }));
          for (const pair of cStr.split(';').map(c => c.trim()).filter(Boolean)) {
            const xVal = parseInt(pair.split(',')[0].split('=')[1].replace(/[^0-9]/g, ''));
            const yVal = parseInt(pair.split(',')[1].split('=')[1].replace(/[^0-9]/g, ''));
            await humanClick(page, box.x + xVal / canvasDims.w * canvasDims.cssW, box.y + yVal / canvasDims.h * canvasDims.cssH);
          }
          await sleep(rand(600, 1000));
          const btn = await frame.$('button[type="submit"]');
          if (btn) await btn.click({ delay: rand(30, 80) });
          await sleep(rand(1500, 2500));
          return !(await frame.$(canvasSel));
        }
        throw new Error('Other kaleidoscope: could not extract puzzle data');
      }
    }

    let b64 = await this._fetchImageB64(frame, imgSrc);
    if (Array.isArray(b64)) b64 = b64[0] || '';

    const rawResult = await this.client.click({
      type: 'base64', body: b64, click: 'oth2',
      textinstructions: 'puzzlekal-' + (typeof puzzle === 'string' ? puzzle : ''),
    }, this.debug);

    const sliderValue = parseInt(String(rawResult).replace(/[^0-9]/g, ''));
    this._log(`Puzzle answer: ${sliderValue}`);

    const sliderEl = await frame.$('.CaptchaSlider');
    let thumbEl = await frame.$('.CaptchaSlider .Thumb');
    if (!sliderEl || !thumbEl) throw new Error('Other puzzle: slider/thumb not found');

    let thumbBox = await thumbEl.boundingBox();
    let sliderBox = await sliderEl.boundingBox();

    if (thumbBox && sliderBox) {
      const thumbOffset = thumbBox.x - sliderBox.x;
      if (thumbOffset > thumbBox.width) {
        this._log('Resetting slider to start position');
        await this._dragSliderToStart(page, thumbEl, sliderBox);
        await sleep(rand(300, 600));
        thumbEl = await frame.$('.CaptchaSlider .Thumb');
        if (!thumbEl) throw new Error('Other puzzle: thumb not found after reset');
        const newSliderEl = await frame.$('.CaptchaSlider');
        if (newSliderEl) sliderBox = await newSliderEl.boundingBox();
      }
    }

    await this._dragSliderToValue(page, frame, thumbEl, sliderValue, sliderBox.width);

    await sleep(rand(800, 1800));

    let btn = await frame.$('button[data-testid="submit"]');
    if (!btn) btn = await frame.$('button[type="submit"]');
    if (btn) {
      this._log('Clicking submit');
      const btnBox = await btn.boundingBox();
      if (btnBox) {
        await page.mouse.move(btnBox.x + btnBox.width / 2 + rand(-5, 5), btnBox.y + btnBox.height / 2 + rand(-3, 3));
        await sleep(rand(150, 400));
      }
      await btn.click({ delay: rand(40, 100) });
    } else {
      this._log('WARNING: submit button not found!');
    }

    await sleep(rand(2500, 4000));
    return !(await frame.$('.AdvancedCaptcha_kaleidoscope'));
  }

  async _solveTextInput(page, frame) {
    const imgEl = await frame.$('.AdvancedCaptcha-View>img');
    if (!imgEl) throw new Error('Other text: image element not found');

    const imgSrc = await imgEl.evaluate(el => el.src);
    let b64 = await this._fetchImageB64(frame, imgSrc);
    if (Array.isArray(b64)) b64 = b64[0] || '';

    const rawResult = await this.client.click({ type: 'base64', body: b64, vernet: '18' }, this.debug);
    if (!rawResult) throw new Error('Other text: no result from API');

    let textInput = await frame.$('.Textinput-Control');
    if (!textInput) textInput = await frame.$('input[type="text"]');
    if (!textInput) throw new Error('Other text: input field not found');

    await textInput.click();
    await sleep(rand(200, 500));
    await textInput.evaluate(el => { el.value = ''; });
    await sleep(rand(100, 300));
    await textInput.type(String(rawResult), { delay: rand(50, 120) });

    await sleep(rand(1500, 2500));
    await page.keyboard.press('Enter');
    await sleep(rand(2500, 3500));
    return !(await frame.$('.AdvancedCaptcha-View>img'));
  }

  async _dragSliderToStart(page, thumb, sliderBox) {
    try { await page.mouse.up(); } catch {}
    await sleep(rand(100, 200));

    const thumbBox = await thumb.boundingBox();
    if (!thumbBox) return;

    const startX = thumbBox.x + thumbBox.width / 2;
    const startY = thumbBox.y + thumbBox.height / 2;
    const targetX = sliderBox.x + thumbBox.width / 2 + 2;
    const distance = startX - targetX;
    if (distance <= 2) return;

    await page.mouse.move(startX, startY);
    await sleep(rand(150, 350));
    await page.mouse.down();
    await sleep(rand(50, 150));

    const steps = Math.max(15, Math.floor(distance / 3) + rand(10, 20));
    for (let i = 0; i <= steps; i++) {
      const progress = i / steps;
      let eased;
      if (progress < 0.5) eased = 4 * progress * progress * progress;
      else eased = 1 - Math.pow(-2 * progress + 2, 3) / 2;
      const cx = startX - distance * eased;
      const cy = startY + (Math.random() - 0.5) * 2;
      await page.mouse.move(cx, cy);
      await sleep(rand(8, 22));
    }

    await sleep(rand(80, 200));
    await page.mouse.up();
    await sleep(rand(200, 500));
  }

  async _dragSliderToValue(page, frame, thumb, targetValue, maxWidth) {
    const box = await thumb.boundingBox();
    if (!box) return;

    try { await page.mouse.up(); } catch {}
    await sleep(rand(100, 200));

    const startX = box.x + box.width / 2;
    const startY = box.y + box.height / 2;

    await page.mouse.move(startX, startY);
    await sleep(rand(200, 500));
    await page.mouse.down();
    await sleep(rand(80, 200));

    const ariaMax = parseInt(await thumb.evaluate(el => el.getAttribute('aria-valuemax') || '100'));
    const ariaMin = parseInt(await thumb.evaluate(el => el.getAttribute('aria-valuemin') || '0'));

    let targetAria = targetValue;
    if (targetAria > ariaMax + 2) {
      const sliderEl = await frame.$('.CaptchaSlider');
      const sliderRect = sliderEl ? await sliderEl.boundingBox() : null;
      const trackWidth = sliderRect ? sliderRect.width : maxWidth;
      const maxDistance = Math.max(1, trackWidth - box.width);
      targetAria = Math.round((targetValue / maxDistance) * (ariaMax - ariaMin) + ariaMin);
    }
    targetAria = Math.max(ariaMin, Math.min(ariaMax, targetAria));

    const stepPx = 4;
    let currentOffset = 0;
    let lastAria = null;
    let lastPauseOffset = 0;
    let yDrift = 0;
    const yDriftDir = Math.random() < 0.5 ? 1 : -1;

    const maxSteps = Math.floor(maxWidth / stepPx) + 60;
    for (let stepI = 0; stepI < maxSteps; stepI++) {
      currentOffset += stepPx + (Math.random() - 0.5) * 1.6;
      const cx = startX + currentOffset;

      yDrift += (Math.random() - 0.5) * 0.8 * yDriftDir;
      yDrift *= 0.90;
      const cy = startY + yDrift + (Math.random() - 0.5) * 2.4;

      await page.mouse.move(cx, cy);

      const nowRaw = await thumb.evaluate(el => el.getAttribute('aria-valuenow'));
      let nowVal = null;
      if (nowRaw !== null) {
        nowVal = parseInt(nowRaw);
        lastAria = nowVal;

        if (nowVal >= targetAria) {
          if (nowVal > targetAria) {
            for (let back = 0; back < 30; back++) {
              currentOffset -= 1;
              const bx = startX + currentOffset;
              const by = startY + (Math.random() - 0.5) * 1.6;
              await page.mouse.move(bx, by);
              await sleep(rand(45, 80));
              const bkRaw = await thumb.evaluate(el => el.getAttribute('aria-valuenow'));
              if (bkRaw !== null && parseInt(bkRaw) <= targetAria) break;
            }
          }
          break;
        }
      }

      let progressToTarget = 0;
      if (ariaMax > ariaMin && targetAria > ariaMin && lastAria !== null) {
        progressToTarget = Math.min(1, Math.max(0, (lastAria - ariaMin) / (targetAria - ariaMin)));
      }

      let speedMs = 45 + Math.floor((1 - Math.sin(progressToTarget * Math.PI * 0.5)) * 90) + rand(0, 55);
      if (Math.random() < 0.06) speedMs += Math.floor(Math.abs((Math.random() - 0.5) * 170));
      await sleep(speedMs);

      if (Math.random() < 0.035 && (currentOffset - lastPauseOffset) > 50) {
        await sleep(rand(30, 80));
        lastPauseOffset = currentOffset;
      }
    }

    if (lastAria === null) {
      const fallbackDist = (targetAria / Math.max(1, ariaMax - ariaMin)) * maxWidth;
      const targetX = startX + Math.max(0, Math.min(maxWidth, fallbackDist));
      await page.mouse.move(targetX, startY);
      await sleep(rand(30, 60));
    }

    await sleep(rand(100, 350));
    await page.mouse.up();
    await sleep(rand(400, 800));
  }
}

module.exports = { OtherSolver };
