async function sleep(ms) {
  return new Promise(resolve => setTimeout(resolve, ms));
}

function rand(min, max) {
  return Math.floor(Math.random() * (max - min + 1)) + min;
}

class SelectorStep {
  constructor(type, value = null) {
    this.type = type;
    this.value = value;
  }
}

function parseSelector(raw) {
  const steps = [];
  const parts = raw.split(/(?=>(?:CSS>|XPATH>|FRAME>))/);
  for (let part of parts) {
    const trimmed = part.replace(/^>+/, '').trim();
    if (!trimmed) continue;
    if (trimmed.startsWith('CSS>')) {
      steps.push(new SelectorStep('css', trimmed.slice(4).trim()));
    } else if (trimmed.startsWith('XPATH>')) {
      steps.push(new SelectorStep('xpath', trimmed.slice(6).trim()));
    } else if (trimmed.startsWith('FRAME>')) {
      steps.push(new SelectorStep('frame'));
    } else {
      steps.push(new SelectorStep('css', trimmed));
    }
  }
  return steps;
}

async function findElement(context, rawSelector) {
  const steps = parseSelector(rawSelector);
  let ctx = context;
  let el = null;
  for (const step of steps) {
    if (step.type === 'frame') {
      if (!el) throw new Error(`FRAME step without prior element in: ${rawSelector}`);
      ctx = await el.contentFrame();
      el = null;
      continue;
    }
    if (step.type === 'css') {
      el = await ctx.$(step.value);
    } else if (step.type === 'xpath') {
      const results = await ctx.$x(step.value);
      el = results.length ? results[0] : null;
    }
    if (!el) return null;
  }
  return el;
}

async function waitForElement(context, rawSelector, timeout = 15000) {
  const deadline = Date.now() + timeout;
  while (Date.now() < deadline) {
    const el = await findElement(context, rawSelector);
    if (el) return el;
    await sleep(300);
  }
  return null;
}

async function screenshotElement(el) {
  const buf = await el.screenshot({ encoding: 'base64' });
  return buf;
}

async function canvasToBase64(pageOrFrame, cssSelector) {
  return await pageOrFrame.evaluate((sel) => {
    const canvas = document.querySelector(sel);
    if (!canvas) return null;
    return canvas.toDataURL('image/png').split(',')[1];
  }, cssSelector);
}

async function humanMove(page, targetX, targetY) {
  const vp = await page.viewport() || { width: 1280, height: 800 };
  const startX = rand(Math.floor(vp.width * 0.3), Math.floor(vp.width * 0.7));
  const startY = rand(Math.floor(vp.height * 0.3), Math.floor(vp.height * 0.7));
  const steps = rand(8, 20);
  const cpX = (startX + targetX) / 2 + (Math.random() - 0.5) * 60;
  const cpY = (startY + targetY) / 2 + (Math.random() - 0.5) * 60;
  for (let i = 1; i <= steps; i++) {
    const t = i / steps;
    const x = (1-t)*(1-t)*startX + 2*(1-t)*t*cpX + t*t*targetX + (Math.random()-0.5)*2;
    const y = (1-t)*(1-t)*startY + 2*(1-t)*t*cpY + t*t*targetY + (Math.random()-0.5)*2;
    await page.mouse.move(x, y);
    await sleep(rand(5, 18));
  }
}

async function humanClick(page, x, y) {
  await humanMove(page, x, y);
  await sleep(rand(80, 200));
  await page.mouse.down();
  await sleep(rand(30, 90));
  await page.mouse.up();
}

async function randomMouseWiggle(page) {
  if (Math.random() > 0.6) return;
  const vp = await page.viewport() || { width: 1280, height: 800 };
  const moves = rand(1, 3);
  for (let i = 0; i < moves; i++) {
    const x = rand(Math.floor(vp.width * 0.15), Math.floor(vp.width * 0.85));
    const y = rand(Math.floor(vp.height * 0.15), Math.floor(vp.height * 0.85));
    await page.mouse.move(x, y, { steps: rand(5, 15) });
    await sleep(rand(50, 400));
  }
}

module.exports = {
  sleep,
  rand,
  SelectorStep,
  parseSelector,
  findElement,
  waitForElement,
  screenshotElement,
  canvasToBase64,
  humanMove,
  humanClick,
  randomMouseWiggle,
};
