const { CaptchaSolverBase } = require('./base_solver');
const { sleep, rand, randomMouseWiggle } = require('./helpers');

class HcaptchaSolver extends CaptchaSolverBase {
  async solve(selector = '') {
    this._log('Solving hCaptcha...');
    await randomMouseWiggle(this.page);
    for (let attempt = 0; attempt < this.attempts; attempt++) {
      this._log(`Attempt ${attempt + 1}/${this.attempts}`);
      try {
        if (await this._attempt()) {
          this._log('hCaptcha solved!');
          return true;
        }
      } catch (e) {
        this._log('Error:', e);
        if (attempt < this.attempts - 1) await sleep(rand(1500, 3000));
      }
    }
    throw new Error('hCaptcha: failed. Error: ERROR_CAPTCHA_UNSOLVABLE');
  }

  async _findFrame(timeout = 10000, selectorFn = null) {
    const deadline = Date.now() + timeout;
    while (Date.now() < deadline) {
      for (const f of this.page.frames()) {
        if (!f.url().includes('hcaptcha.com')) continue;
        try {
          if (!selectorFn || await selectorFn(f)) return f;
        } catch {}
      }
      await sleep(300);
    }
    return null;
  }

  async _isSolved(checkboxFrame) {
    try {
      return await checkboxFrame.evaluate(() => {
        const cb = document.querySelector('#checkbox');
        if (cb && cb.getAttribute('aria-checked') === 'true') return true;
        const div = document.querySelector('div.check');
        return !!(div && div.style.display === 'block');
      });
    } catch { return false; }
  }

  async _switchLanguage(challengeFrame) {
    try {
      const lang = await challengeFrame.evaluate(
        () => (document.documentElement.lang || navigator.language || '').toLowerCase()
      );
      if (lang.startsWith('en')) return;

      const isBbox = !!(await challengeFrame.$('.bounding-box-example'));
      const isHeader = !!(await challengeFrame.$('.challenge-header'));
      const is9grid = (await challengeFrame.$$('.task-image .image')).length === 9;
      const isMulti = !!(await challengeFrame.$('.task-answers'));

      if (isBbox || (isHeader && !is9grid) || isMulti || (!is9grid)) {
        const btn = await challengeFrame.$('.display-language.button');
        if (btn) {
          await challengeFrame.evaluate(el => el.click(), btn);
          await sleep(200);
        }
      }

      const opt = await challengeFrame.$('.language-selector .option:nth-child(23)');
      if (opt) {
        await challengeFrame.evaluate(el => el.click(), opt);
        await sleep(300);
      }

      await sleep(500);
      this._log('hCaptcha: switched language to English');
    } catch (e) {
      this._log('hCaptcha: language switch error:', e);
    }
  }

  async _gatherData(challengeFrame) {
    const GET_BG_JS = `(el) => (el.style.backgroundImage || '').replace(/url\\("|"\\)/g, '')`;
    const FETCH_B64_JS = `async (url) => {
      if (!url || url === 'none' || url === '') return null;
      try {
        const r = await fetch(url);
        if (!r.ok) return null;
        const blob = await r.blob();
        return await new Promise(res => {
          const rd = new FileReader();
          rd.readAsDataURL(blob);
          rd.onloadend = () => res((rd.result||'').replace(/^data:image\\/(png|jpeg);base64,/, ''));
          rd.onerror = () => res(null);
        });
      } catch(e) { return null; }
    }`;

    for (let attempt = 0; attempt < 40; attempt++) {
      await sleep(500);
      try {
        const prompt = await challengeFrame.evaluate(() => {
          const el = document.querySelector('h2.prompt-text') || document.querySelector('.prompt-text');
          return el ? (el.innerText || el.textContent || '').trim().replace(/\s+/g, ' ') : '';
        });
        if (!prompt) continue;

        const isBbox = !!(await challengeFrame.$('.bounding-box-example'));
        const isHeader = !!(await challengeFrame.$('.challenge-header'));
        const isMulti = !!(await challengeFrame.$('.task-answers'));
        const taskImgs = await challengeFrame.$$('.task-image .image');
        const is9grid = taskImgs.length === 9;

        const images = {};
        const examples = [];
        let choices = [];
        let captchaType;

        if (is9grid || isMulti) {
          let ok = true;
          for (let idx = 0; idx < taskImgs.length; idx++) {
            try {
              const buf = await taskImgs[idx].screenshot({ encoding: 'base64' });
              images[idx] = buf;
            } catch { ok = false; break; }
          }
          if (!ok || !Object.keys(images).length) continue;
          const exEls = await challengeFrame.$$('.example-image .image');
          for (const exEl of exEls) {
            try {
              const buf = await exEl.screenshot({ encoding: 'base64' });
              examples.push(buf);
            } catch {}
          }
          if (isMulti) {
            choices = await challengeFrame.evaluate(
              () => Array.from(document.querySelectorAll('.answer-text')).map(e => (e.outerText || e.textContent || '').trim()).filter(Boolean)
            );
          }
          captchaType = isMulti ? 'multi' : 'grid';

        } else if (isBbox || (isHeader && !is9grid)) {
          const b64 = await challengeFrame.evaluate(() => {
            const canvas = document.querySelector('canvas');
            if (!canvas) return null;
            try {
              const d = canvas.getContext('2d', { willReadFrequently: true }).getImageData(0, 0, canvas.width, canvas.height).data;
              let mono = true;
              for (let i = 0; i < d.length; i += 4) if (d[i] !== d[i+1] || d[i+1] !== d[i+2]) { mono = false; break; }
              if (mono) return null;
              const sw = parseInt(canvas.style.width, 10) || canvas.width;
              const sh = parseInt(canvas.style.height, 10) || canvas.height;
              if (sw <= 0 || sh <= 0) return null;
              const sc = Math.min(sw / canvas.width, sh / canvas.height);
              const tw = canvas.width * sc, th = canvas.height * sc;
              const tmp = document.createElement('canvas');
              tmp.width = tw; tmp.height = th;
              tmp.getContext('2d').drawImage(canvas, 0, 0, canvas.width, canvas.height, 0, 0, tw, th);
              return tmp.toDataURL('image/jpeg').replace(/^data:image\/(png|jpeg);base64,/, '');
            } catch { return null; }
          });
          if (!b64 || b64.length * 3 / 4 / 1024 < 10) continue;
          images[0] = b64;
          const exEls = await challengeFrame.$$('.example-image .image');
          for (const exEl of exEls) {
            const url = await challengeFrame.evaluate(GET_BG_JS, exEl);
            if (url && url !== 'none') {
              const b64ex = await challengeFrame.evaluate(FETCH_B64_JS, url);
              if (b64ex) examples.push(b64ex);
            }
          }
          captchaType = (isHeader && !isBbox) ? 'bboxdd' : 'bbox';

        } else if (taskImgs.length) {
          let ok = true;
          for (let idx = 0; idx < taskImgs.length; idx++) {
            try {
              const buf = await taskImgs[idx].screenshot({ encoding: 'base64' });
              images[idx] = buf;
            } catch { ok = false; break; }
          }
          if (!ok || !Object.keys(images).length) continue;
          captchaType = 'grid';
        } else {
          continue;
        }

        return { prompt, images, examples, choices, type: captchaType, tiles: taskImgs };
      } catch (e) {
        this._log('hCaptcha gather_data error:', e);
      }
    }
    return null;
  }

  async _applyAnswer(challengeFrame, data, answer) {
    answer = answer.trim();
    const captchaType = data.type || 'grid';

    if (answer === 'notpic') {
      await challengeFrame.evaluate(() => { const b = document.querySelector('.button-submit'); b && b.click(); });
      await sleep(1200);
      return 'applied';
    }

    if (answer.startsWith('coordinates:')) {
      const parts = answer.replace('coordinates:', '').split(';').map(s => s.trim()).filter(Boolean);
      await challengeFrame.evaluate(async (parts) => {
        const canvas = document.querySelector('canvas');
        if (!canvas) return;
        const rect = canvas.getBoundingClientRect();
        for (const part of parts) {
          const xm = part.match(/x=(\d+)/), ym = part.match(/y=(\d+)/);
          if (!xm || !ym) continue;
          const cx = parseInt(xm[1]) + rect.left, cy = parseInt(ym[1]) + rect.top;
          canvas.dispatchEvent(new MouseEvent('mousedown', { clientX: cx, clientY: cy, bubbles: true, button: 0 }));
          canvas.dispatchEvent(new MouseEvent('mouseup', { clientX: cx, clientY: cy, bubbles: true, button: 0 }));
          canvas.dispatchEvent(new MouseEvent('click', { clientX: cx, clientY: cy, bubbles: true, button: 0 }));
          await new Promise(r => setTimeout(r, 200 + 150 * Math.random()));
        }
      }, parts);
      await sleep(500);
      await challengeFrame.evaluate(() => { const b = document.querySelector('.button-submit'); b && b.click(); });
      await sleep(2500);
      return 'applied';
    }

    if (answer.startsWith('move:')) {
      const pairs = [];
      for (const part of answer.replace('move:', '').split(';')) {
        const xm = part.match(/x=(\d+)/);
        const ym = part.match(/y=(\d+)/);
        pairs.push([xm ? parseInt(xm[1]) : 0, ym ? parseInt(ym[1]) : 0]);
      }
      if (pairs.length >= 2) {
        await challengeFrame.evaluate(async (pairs) => {
          const canvas = document.querySelector('canvas');
          if (!canvas || pairs.length < 2) return;
          const rect = canvas.getBoundingClientRect();
          for (let i = 0; i + 1 < pairs.length; i += 2) {
            const ax = pairs[i][0] + rect.left, ay = pairs[i][1] + rect.top;
            const bx = pairs[i+1][0] + rect.left, by = pairs[i+1][1] + rect.top;
            canvas.dispatchEvent(new MouseEvent('mousedown', { clientX: ax, clientY: ay, bubbles: true, button: 0, buttons: 1 }));
            await new Promise(r => setTimeout(r, 100));
            for (let s = 1; s <= 15; s++) {
              canvas.dispatchEvent(new MouseEvent('mousemove', {
                clientX: ax + (bx - ax) / 15 * s, clientY: ay + (by - ay) / 15 * s, bubbles: true, buttons: 1
              }));
              await new Promise(r => setTimeout(r, 30));
            }
            canvas.dispatchEvent(new MouseEvent('mouseup', { clientX: bx, clientY: by, bubbles: true, button: 0 }));
            if (i + 3 < pairs.length) await new Promise(r => setTimeout(r, 1500));
          }
        }, pairs);
      }
      await sleep(500);
      await challengeFrame.evaluate(() => { const b = document.querySelector('.button-submit'); b && b.click(); });
      await sleep(2500);
      return 'applied';
    }

    if (captchaType === 'multi') {
      const texts = answer.split(',').map(t => t.trim());
      await challengeFrame.evaluate((texts) => {
        function fire(el) {
          ['mouseover','mousedown','mouseup','click'].forEach(type => {
            const e = document.createEvent('MouseEvents'); e.initEvent(type, true, false); el.dispatchEvent(e);
          });
        }
        const els = Array.from(document.querySelectorAll('.answer-text'));
        for (const t of texts) { const el = els.find(e => (e.outerText || '').trim() === t); if (el) fire(el); }
      }, texts);
      await sleep(500);
      await challengeFrame.evaluate(() => { const b = document.querySelector('.button-submit'); b && b.click(); });
      await sleep(2500);
      return 'applied';
    }

    if (captchaType === 'bbox') {
      const nums = answer.split(/[;:,]/).filter(n => n.trim().match(/^\d+$/)).map(Number);
      if (nums.length >= 2) {
        await challengeFrame.evaluate((coords) => {
          const canvas = document.querySelector('canvas');
          if (!canvas) return;
          const rect = canvas.getBoundingClientRect();
          const opts = { clientX: coords[0] + rect.left, clientY: coords[1] + rect.top, bubbles: true };
          ['mouseover','mousedown','mouseup','click'].forEach(t => canvas.dispatchEvent(new MouseEvent(t, opts)));
        }, nums.slice(0, 2));
      }
      return 'applied';
    }

    if (captchaType === 'bboxdd') {
      await sleep(500);
      await challengeFrame.evaluate(() => { const b = document.querySelector('.button-submit'); b && b.click(); });
      await sleep(2500);
      return 'applied';
    }

    const tileIndices = this.client.parseTileIndices(answer);
    if (tileIndices.length) {
      const tiles = data.tiles || [];
      for (const idx of tileIndices) {
        if (idx < tiles.length) {
          await sleep(rand(150, 350));
          await challengeFrame.evaluate((el) => {
            ['mouseover','mousedown','mouseup','click'].forEach(type => {
              const e = document.createEvent('MouseEvents'); e.initEvent(type, true, false); el.dispatchEvent(e);
            });
          }, tiles[idx]);
        }
      }
    }
    await sleep(500);
    await challengeFrame.evaluate(() => { const b = document.querySelector('.button-submit'); b && b.click(); });
    await sleep(2500);
    return 'applied';
  }

  async _attempt() {
    const page = this.page;

    const hasCheckbox = async (f) => !!(await f.$('#checkbox, div.check'));

    const checkboxFrame = await this._findFrame(10000, hasCheckbox);
    if (!checkboxFrame) {
      this._log('hCaptcha: checkbox frame not found');
      return false;
    }

    this._log('hCaptcha: checkbox frame found:', checkboxFrame.url().slice(0, 80));

    if (await this._isSolved(checkboxFrame)) {
      this._log('hCaptcha: already solved');
      return true;
    }

    this._log('Clicking hCaptcha checkbox...');
    await sleep(rand(300, 700));
    await checkboxFrame.evaluate(() => {
      const cb = document.querySelector('#checkbox') || document.querySelector('div.check');
      if (cb) cb.click();
    });
    await sleep(rand(2000, 3500));

    if (await this._isSolved(checkboxFrame)) {
      this._log('hCaptcha: solved without image challenge!');
      return true;
    }

    const hasPrompt = async (f) => !!(await f.$('h2.prompt-text, .prompt-text'));
    const challengeFrame = await this._findFrame(15000, hasPrompt);
    if (!challengeFrame) {
      this._log('hCaptcha: challenge frame not found, checking if solved...');
      return await this._isSolved(checkboxFrame);
    }

    this._log('hCaptcha: challenge frame found:', challengeFrame.url().slice(0, 80));

    let failCount = 0;
    let lastTaskKey = null;

    for (let iteration = 0; iteration < 15; iteration++) {
      await sleep(rand(300, 600));

      if (await this._isSolved(checkboxFrame)) {
        this._log('hCaptcha: solved!');
        return true;
      }

      if (!(await challengeFrame.$('h2.prompt-text, .prompt-text'))) {
        await sleep(1000);
        if (await this._isSolved(checkboxFrame)) return true;
        continue;
      }

      await this._switchLanguage(challengeFrame);

      this._log(`hCaptcha: iteration ${iteration + 1} — gathering data...`);
      const data = await this._gatherData(challengeFrame);

      if (!data) {
        this._log('hCaptcha: failed to gather data');
        failCount++;
        if (failCount >= 2) {
          this._log('hCaptcha: refreshing challenge');
          await challengeFrame.evaluate(() => {
            const btn = document.querySelector('.refresh.button') ||
                        document.querySelector('.button-submit') ||
                        document.querySelector('[aria-label="Refresh"]');
            if (btn) btn.click();
          });
          failCount = 0;
          lastTaskKey = null;
        }
        await sleep(rand(800, 1500));
        continue;
      }

      this._log(`hCaptcha: type=${data.type}, prompt=${data.prompt}, imgs=${Object.keys(data.images).length}`);

      const taskKey = JSON.stringify({ p: data.prompt, k: Object.keys(data.images) });
      if (taskKey === lastTaskKey) await sleep(800);
      lastTaskKey = taskKey;

      const sortedKeys = Object.keys(data.images).sort((a, b) => a - b);
      const imageList = sortedKeys.map(k => data.images[k]);
      const body = imageList.length === 1 ? imageList[0] : imageList;

      const payload = {
        type: 'base64',
        textinstructions: data.prompt,
      };

      if (Array.isArray(body) && body.length > 1) {
        payload.click = 'hcap2';
        body.forEach((b, i) => { payload[`body${i}`] = b; });
      } else {
        payload.click = 'hcap';
        payload.body = Array.isArray(body) ? body[0] : body;
      }

      if (data.examples && data.examples.length) payload.sizex = 10;
      if (data.choices && data.choices.length) payload.textinstructions += '|||' + data.choices.join(',');

      const answerResult = await this.client.click(payload, this.debug);
      this._log('hCaptcha: API answer:', answerResult);

      if (!answerResult || answerResult.includes('ERROR')) {
        this._log('hCaptcha: no answer from API, fail count:', failCount + 1);
        failCount++;
        if (failCount >= 2) {
          this._log('hCaptcha: refreshing challenge');
          await challengeFrame.evaluate(() => {
            const btn = document.querySelector('.refresh.button') ||
                        document.querySelector('.button-submit') ||
                        document.querySelector('[aria-label="Refresh"]');
            if (btn) btn.click();
          });
          failCount = 0;
          lastTaskKey = null;
        }
        await sleep(rand(800, 1500));
        continue;
      }

      const applyResult = await this._applyAnswer(challengeFrame, data, answerResult);
      this._log('hCaptcha: apply result:', applyResult);

      if (applyResult === 'solved') return true;

      failCount = 0;
      lastTaskKey = null;

      await sleep(rand(500, 1000));
      if (await this._isSolved(checkboxFrame)) return true;
    }

    return await this._isSolved(checkboxFrame);
  }
}

module.exports = { HcaptchaSolver };
