const { CaptchaSolverBase } = require('./base_solver');
const { sleep, rand, waitForElement, screenshotElement, humanClick, randomMouseWiggle } = require('./helpers');

class GeetestSolver extends CaptchaSolverBase {
  async solve(selector = '') {
    this._log('Solving GeeTest...');
    await randomMouseWiggle(this.page);
    for (let i = 0; i < 30; i++) {
      const el = await this.page.$(
        "[class*='geetest_btn_click'], [class*='geetest_radar_btn'], " +
        "[class*='geetest_holder'], [class*='geetest_box_wrap']"
      );
      if (el) { await sleep(rand(300, 600)); break; }
      await sleep(300);
    }
    for (let attempt = 0; attempt < this.attempts; attempt++) {
      this._log(`Attempt ${attempt + 1}/${this.attempts}`);
      try {
        if (await this._attempt()) {
          this._log('GeeTest solved!');
          return true;
        }
      } catch (e) {
        this._log('Error:', e);
        if (attempt < this.attempts - 1) await sleep(rand(1500, 3000));
      }
    }
    throw new Error('GeeTest: failed. Error: ERROR_CAPTCHA_UNSOLVABLE');
  }

  async _detectType(page) {
    if (await page.$("div[class*='geetest_holder'][class*='geetest_ant'] :first-child[class*='geetest_canvas_img'] :first-child[class*='geetest_slicebg']"))
      return 'GEETEST_VERSION_3_SLIDER';
    if (await page.$("div[class*='geetest_silver'][style*='block'] [class*='geetest_item_wrap']"))
      return 'GEETEST_VERSION_3_CLICK';
    if (await page.$("div[class*='geetest_box'][style*='block'] :first-child[class*='geetest_track']"))
      return 'GEETEST_VERSION_4_SLIDER';
    if (await page.$("[class*='geetest_box'][style*='block'] :first-child[class*='geetest_item-0-0']"))
      return 'GEETEST_VERSION_4_SWAP_ITEMS';

    const v4Click = await page.evaluate(() => {
      const el = document.querySelector("[class*='geetest_box_wrap'][style*='block'] :first-child[class*='geetest_box']");
      if (!el) return false;
      const wrap = el.closest("[class*='geetest_box_wrap']");
      if (!wrap) return false;
      if (wrap.querySelector("[class*='geetest_track']")) return false;
      if (wrap.querySelector("[class*='geetest_item-0-0']")) return false;
      return true;
    });
    if (v4Click) return 'GEETEST_VERSION_4_CLICK';

    if (await page.$(
      "[class*='geetest_radar_btn'], [class*='geetest_radar_tip'], " +
      "[class*='geetest_holder']:not([style*='none'])"
    )) return 'GEETEST_CHECKBOX_V3';

    if (await page.$(
      "[class*='geetest_btn_click'], " +
      "[class*='geetest_box_wrap']:not([style*='none']), " +
      "[class*='geetest_checkbox']:not([style*='none'])"
    )) return 'GEETEST_CHECKBOX_V4';

    return 'UNKNOWN';
  }

  static CHECKBOX_SELECTORS = [
    '.geetest_radar_btn',
    "[class*='geetest_radar_btn']",
    '.geetest_btn',
    "[class*='geetest_checkbox']",
    "[class*='geetest_btn_click']",
    "[class*='geetest_box'] [class*='geetest_btn']",
    "[class*='geetest_holder'] [class*='geetest_icon']",
    "[class*='geetest_radar_tip']",
    "[class*='geetest_radar']",
  ];

  async _clickCheckbox(page) {
    const sel = this.selector;
    if (sel) {
      this._log('GeeTest: trying user selector:', sel);
      const css = sel.replace(/^(>?CSS>\s*)+/, '').trim();
      const btn = css ? await page.$(css) : null;
      if (btn) {
        await btn.evaluate(el => el.scrollIntoViewIfNeeded());
        await sleep(rand(200, 500));
        await btn.click({ delay: rand(40, 100) });
        this._log('GeeTest: clicked user selector button');
        return true;
      } else {
        this._log('GeeTest: user selector element not found on page:', css);
      }
    }
    for (const css of GeetestSolver.CHECKBOX_SELECTORS) {
      const btn = await page.$(css);
      if (btn) {
        const visible = await btn.evaluate(el => {
          const s = window.getComputedStyle(el);
          return s.display !== 'none' && s.visibility !== 'hidden' && el.offsetWidth > 0;
        });
        if (!visible) continue;
        this._log('GeeTest: found checkbox via selector:', css);
        await btn.evaluate(el => el.scrollIntoViewIfNeeded());
        await sleep(rand(200, 500));
        await btn.click({ delay: rand(40, 100) });
        this._log('GeeTest: clicked checkbox button');
        return true;
      }
    }
    this._log('GeeTest: no checkbox/button found to click');
    return false;
  }

  async _attempt() {
    const page = this.page;
    let captchaType = await this._detectType(page);

    if (captchaType === 'UNKNOWN') {
      await this._clickCheckbox(page);
      this._log('GeeTest: waiting for captcha window to appear...');
      for (let i = 0; i < 50; i++) {
        captchaType = await this._detectType(page);
        if (!['UNKNOWN', 'GEETEST_CHECKBOX_V3', 'GEETEST_CHECKBOX_V4'].includes(captchaType)) break;
        await sleep(300);
      }
      if (['UNKNOWN', 'GEETEST_CHECKBOX_V3', 'GEETEST_CHECKBOX_V4'].includes(captchaType))
        throw new Error('GeeTest: captcha window not found or unknown type');
    }

    if (['GEETEST_CHECKBOX_V3', 'GEETEST_CHECKBOX_V4'].includes(captchaType)) {
      this._log('GeeTest: checkbox detected, clicking to open challenge...');
      const clicked = await this._clickCheckbox(page);
      if (!clicked) throw new Error('GeeTest: checkbox found but could not click it');
      for (let i = 0; i < 50; i++) {
        captchaType = await this._detectType(page);
        if (!['UNKNOWN', 'GEETEST_CHECKBOX_V3', 'GEETEST_CHECKBOX_V4'].includes(captchaType)) break;
        await sleep(300);
      }
      if (['UNKNOWN', 'GEETEST_CHECKBOX_V3', 'GEETEST_CHECKBOX_V4'].includes(captchaType)) {
        if (await this._checkSuccess(page)) return true;
        throw new Error('GeeTest: captcha window not found after clicking checkbox');
      }
    }

    this._log('GeeTest detected type:', captchaType);

    if (captchaType === 'GEETEST_VERSION_3_SLIDER') return await this._v3Slider(page);
    if (captchaType === 'GEETEST_VERSION_4_SLIDER') return await this._v4Slider(page);
    if (captchaType === 'GEETEST_VERSION_3_CLICK') return await this._v3Click(page);
    if (captchaType === 'GEETEST_VERSION_4_CLICK') return await this._v4Click(page);
    if (captchaType === 'GEETEST_VERSION_4_SWAP_ITEMS') return await this._v4Swap(page);
    throw new Error(`GeeTest: unknown type "${captchaType}"`);
  }

  async _v3Slider(page) {
    const canvasSel = "div[class*='geetest_holder'][class*='geetest_ant'] :first-child[class*='geetest_canvas_bg']";
    const canvasEl = await waitForElement(page, `>CSS> ${canvasSel}`, 10000);
    if (!canvasEl) throw new Error('GeeTest V3 Slider: canvas not found');

    for (let w = 0; w < 20; w++) {
      const hasContent = await page.evaluate((sel) => {
        const c = document.querySelector(sel);
        if (!c || c.tagName !== 'CANVAS' || !c.width || !c.height) return false;
        try {
          const ctx = c.getContext('2d');
          const d = ctx.getImageData(0, 0, Math.min(c.width, 50), Math.min(c.height, 50));
          return d.data.some(v => v !== 0);
        } catch { return false; }
      }, canvasSel);
      if (hasContent) break;
      await sleep(500);
    }

    let bgB64 = await page.evaluate((sel) => {
      const c = document.querySelector(sel);
      if (!c || c.tagName !== 'CANVAS') return null;
      return c.toDataURL('image/png').replace(/^data:image[/](png|jpg|jpeg);base64,/, '');
    }, canvasSel);
    if (!bgB64) bgB64 = await screenshotElement(canvasEl);

    const dims = await page.evaluate((sel) => {
      const c = document.querySelector(sel);
      if (!c) return { sent: 0, disp: 0 };
      return { sent: c.width || 0, disp: c.offsetWidth || c.clientWidth || c.getBoundingClientRect().width || 0 };
    }, canvasSel);
    const sentImgW = dims.sent || 260;
    const displayW = dims.disp || sentImgW;

    const rawResult = await this.client.click({
      type: 'base64', click: 'geetest', body: bgB64, textinstructions: 'slider',
    }, this.debug);
    if (!rawResult) throw new Error('GeeTest V3 Slider: empty API result');
    this._log('GeeTest V3 Slider raw result:', rawResult);

    const nums = this.client.parseRawPixelCoords(rawResult);
    if (nums.length < 2) throw new Error(`GeeTest V3 Slider: bad result: ${rawResult}`);

    const answerX = nums[0];
    let dragPx;
    if (sentImgW === displayW || Math.abs(sentImgW - displayW) < 5) {
      dragPx = answerX;
    } else {
      dragPx = Math.round(answerX / sentImgW * displayW);
    }
    this._log(`GeeTest V3 Slider: answer_x=${answerX} sent_w=${sentImgW} display_w=${displayW} drag_px=${dragPx}`);

    const btnEl = await page.$("[class*='geetest_slider_button']");
    if (!btnEl) throw new Error('GeeTest V3 Slider: slider button not found');

    const btnBox = await btnEl.boundingBox();
    const startX = btnBox.x + btnBox.width / 2;
    const startY = btnBox.y + btnBox.height / 2;

    await this._dragPrecise(page, startX, startY, startX + dragPx, startY);
    await sleep(rand(2000, 3000));

    if (await this._checkSolvedAfterDrag(page)) return true;

    await sleep(1000);

    const resetTip = await page.$("[class*='geetest_reset_tip_content']");
    if (resetTip) {
      this._log('GeeTest V3 Slider: error shown, clicking reset');
      await resetTip.click();
      await sleep(rand(1500, 2500));
      return false;
    }

    const sliderReset = await page.evaluate(() => {
      const btn = document.querySelector("[class*='geetest_slider_button']");
      if (!btn) return false;
      const rect = btn.getBoundingClientRect();
      const track = btn.closest("[class*='geetest_slider']");
      if (!track) return rect.left < 50;
      const trackRect = track.getBoundingClientRect();
      return (rect.left - trackRect.left) < 50;
    });

    if (sliderReset) {
      this._log('GeeTest V3 Slider: slider reset, wrong answer');
      const refreshEl = await page.$("a[class*='geetest_refresh']");
      if (refreshEl) {
        await refreshEl.click();
        await sleep(rand(1500, 2500));
      }
      return false;
    }

    await sleep(1000);
    if (await this._checkSolvedAfterDrag(page)) return true;

    this._log('GeeTest V3 Slider: no success detected, clicking refresh');
    const refreshEl = await page.$("a[class*='geetest_refresh'], [class*='geetest_reset_tip_content']");
    if (refreshEl) {
      await refreshEl.click();
      await sleep(rand(1500, 2500));
    }
    return false;
  }

  async _v4Slider(page) {
    const btnSel = "[class*='geetest_box'][style*='block'] [class*='geetest_btn']";
    const btnEl = await waitForElement(page, `>CSS> ${btnSel}`, 10000);
    if (!btnEl) throw new Error('GeeTest V4 Slider: btn not found');

    for (let i = 0; i < 20; i++) {
      const hasBg = await page.evaluate(() => {
        const box = document.querySelector("[class*='geetest_box'][style*='block']");
        if (!box) return false;
        for (const el of box.querySelectorAll("*")) {
          const bg = window.getComputedStyle(el).backgroundImage;
          if (bg && bg !== 'none' && bg.includes('url') && !bg.includes('gradient')) return true;
        }
        const cv = box.querySelector("canvas");
        return !!(cv && cv.width > 0);
      });
      if (hasBg) break;
      await sleep(300);
    }

    const geo = await page.evaluate(() => {
      const box = document.querySelector("[class*='geetest_box'][style*='block']");
      if (!box) return {};
      const result = {};
      const btn = box.querySelector("[class*='geetest_btn']");
      if (btn) { result.btnW = btn.getBoundingClientRect().width; result.btnH = btn.getBoundingClientRect().height; }
      const track = box.querySelector("[class*='geetest_track']");
      if (track) result.trackW = track.getBoundingClientRect().width;
      const canvas = box.querySelector("canvas");
      if (canvas && canvas.width > 0) {
        result.canvasW = canvas.width; result.canvasH = canvas.height;
        result.canvasDisplayW = canvas.offsetWidth || canvas.clientWidth || canvas.getBoundingClientRect().width;
        try { result.canvasB64 = canvas.toDataURL('image/png').split(',')[1]; } catch {}
      }
      for (const el of box.querySelectorAll("*")) {
        const bg = window.getComputedStyle(el).backgroundImage;
        if (bg && bg !== 'none' && bg.includes('url') && !bg.includes('gradient')) {
          const url = bg.replace(/^url\(["']?/, '').replace(/["']?\)$/, '');
          const r = el.getBoundingClientRect();
          if (r.width > 100) { result.bgUrl = url; result.bgElemW = r.width; result.bgElemH = r.height; break; }
        }
      }
      return result;
    });
    this._log('GeeTest V4 Slider geo:', Object.fromEntries(Object.entries(geo).filter(([k]) => k !== 'canvasB64')));

    let bgB64 = geo.canvasB64 || null;
    let sentImgW = geo.canvasW || 0;
    let displayW = geo.canvasDisplayW || 0;

    if (!bgB64 && geo.bgUrl) {
      this._log('GeeTest V4 Slider: fetching bg from:', geo.bgUrl.slice(0, 80));
      try {
        bgB64 = await this._urlToBase64(geo.bgUrl);
        const imgSize = await page.evaluate((url) => new Promise((resolve) => {
          const img = new Image();
          img.onload = () => resolve({ w: img.naturalWidth, h: img.naturalHeight });
          img.onerror = () => resolve({ w: 0, h: 0 });
          img.src = url;
        }), geo.bgUrl);
        sentImgW = imgSize.w || 0;
        displayW = geo.bgElemW || 0;
      } catch (e) {
        this._log('GeeTest V4 Slider: bg fetch failed:', e);
      }
    }

    if (!bgB64) {
      for (const sel of [
        "[class*='geetest_box'][style*='block'] [class*='geetest_window']",
        "[class*='geetest_box'][style*='block']",
      ]) {
        const bgEl = await page.$(sel);
        if (bgEl) {
          bgB64 = await screenshotElement(bgEl);
          const ssBox = await bgEl.boundingBox();
          if (ssBox) { sentImgW = Math.floor(ssBox.width); displayW = Math.floor(ssBox.width); }
          this._log('GeeTest V4 Slider: screenshot fallback via:', sel);
          break;
        }
      }
    }

    if (!bgB64) throw new Error('GeeTest V4 Slider: background image not found');

    const rawResult = await this.client.click({
      type: 'base64', click: 'geetest', body: bgB64, textinstructions: 'slider',
    }, this.debug);
    if (!rawResult) throw new Error('GeeTest V4 Slider: empty API result');
    this._log('GeeTest V4 Slider raw result:', rawResult);

    const nums = this.client.parseRawPixelCoords(rawResult);
    if (!nums.length) throw new Error(`GeeTest V4 Slider: bad result: ${rawResult}`);

    const answerX = nums[0];
    const dragPx = sentImgW > 0 && displayW > 0 ? Math.round(answerX / sentImgW * displayW) : answerX;
    this._log(`GeeTest V4 Slider: answer_x=${answerX} sent_w=${sentImgW} display_w=${displayW} drag_px=${dragPx}`);

    let btnBox = await btnEl.boundingBox();
    if (!btnBox) {
      await btnEl.evaluate(el => el.scrollIntoViewIfNeeded());
      await sleep(300);
      btnBox = await btnEl.boundingBox();
    }
    if (!btnBox) throw new Error('GeeTest V4 Slider: cannot get btn bounding box');

    const startX = btnBox.x + btnBox.width / 2;
    const startY = btnBox.y + btnBox.height / 2;

    await this._dragPrecise(page, startX, startY, startX + dragPx, startY);
    await sleep(rand(2000, 3000));

    if (await this._checkSolvedAfterDrag(page)) return true;

    await sleep(1000);

    const resetTip = await page.$("[class*='geetest_reset_tip_content']");
    if (resetTip) {
      this._log('GeeTest V4 Slider: error shown, clicking reset');
      await resetTip.click();
      await sleep(rand(1500, 2500));
      return false;
    }

    const errorEl = await page.$(
      "div[class*='geetest_radar_error'][style*='width: 300'] [class*='geetest_reset_tip_content']," +
      "div[class*='geetest_panel_error'][style*='block'] [class*='geetest_panel_error_content']," +
      "[class*='geetest_box'][style*='block'] [class*='geetest_reset']," +
      "[class*='geetest_result_tips'][class*='geetest_fail'][class*='geetest_showResult']"
    );
    if (errorEl) {
      this._log('GeeTest V4 Slider: wrong answer, clicking error to retry...');
      await errorEl.click();
      await sleep(rand(1000, 2000));
      return false;
    }

    return await this._checkSolvedAfterDrag(page);
  }

  async _v3Click(page) {
    const containerSel = "div[class*='geetest_silver'][style*='block']";
    const containerEl = await waitForElement(page, `>CSS> ${containerSel}`, 10000);
    if (!containerEl) throw new Error('GeeTest V3 Click: container not found');

    const b64 = await screenshotElement(containerEl);
    const containerBox = await containerEl.boundingBox();

    const rawResult = await this.client.click({
      type: 'base64', click: 'geetest', body: b64, textinstructions: 'geetestv2',
    }, this.debug);
    this._log('GeeTest V3 Click raw result:', rawResult);
    if (!rawResult || rawResult.includes('ERROR'))
      throw new Error(`GeeTest V3 Click: bad API result: ${rawResult}`);

    const coordStr = rawResult.includes('coordinates:') ? rawResult.split('coordinates:')[1] : rawResult;
    const pairs = coordStr.split(';').map(p => p.trim()).filter(Boolean);

    for (const pair of pairs) {
      let xVal = 0, yVal = 0;
      for (const part of pair.split(',')) {
        const t = part.trim();
        if (t.startsWith('x=')) xVal = parseInt(t.slice(2).replace(/[^0-9]/g, ''));
        else if (t.startsWith('y=')) yVal = parseInt(t.slice(2).replace(/[^0-9]/g, ''));
      }
      await humanClick(page, containerBox.x + xVal, containerBox.y + yVal);
      await sleep(rand(200, 500));
    }

    await sleep(rand(300, 600));
    const submitEl = await page.$("div[class*='geetest_silver'][style*='block'] [class*='geetest_commit_tip']");
    if (submitEl) { await submitEl.click({ delay: rand(30, 80) }); await sleep(rand(1000, 2000)); }

    return await this._checkSuccess(page);
  }

  async _v4Click(page) {
    const containerSel = "[class*='geetest_box_wrap'][style*='block'] :first-child[class*='geetest_box']";
    const containerEl = await waitForElement(page, `>CSS> ${containerSel}`, 10000);
    if (!containerEl) throw new Error('GeeTest V4 Click: container not found');

    const b64 = await screenshotElement(containerEl);
    const containerBox = await containerEl.boundingBox();

    const submitEl = await page.$("[class*='geetest_commit_tip'], [class*='geetest_submit']");
    const textInstructions = submitEl ? 'geetesticonhand' : 'geetesticonv4';

    const rawResult = await this.client.click({
      type: 'base64', click: 'geetest', body: b64, textinstructions: textInstructions,
    }, this.debug);
    this._log('GeeTest V4 Click raw result:', rawResult);
    if (!rawResult || rawResult.includes('ERROR'))
      throw new Error(`GeeTest V4 Click: bad API result: ${rawResult}`);

    const coordStr = rawResult.includes('coordinates:') ? rawResult.split('coordinates:')[1] : rawResult;
    const pairs = coordStr.split(';').map(p => p.trim()).filter(Boolean);

    for (const pair of pairs) {
      let xVal = 0, yVal = 0;
      for (const part of pair.split(',')) {
        const t = part.trim();
        if (t.startsWith('x=')) xVal = parseInt(t.slice(2).replace(/[^0-9]/g, ''));
        else if (t.startsWith('y=')) yVal = parseInt(t.slice(2).replace(/[^0-9]/g, ''));
      }
      await humanClick(page, containerBox.x + xVal, containerBox.y + yVal);
      await sleep(rand(200, 500));
    }

    await sleep(rand(300, 600));
    if (submitEl) { await submitEl.click({ delay: rand(30, 80) }); await sleep(rand(1000, 2000)); }

    return await this._checkSuccess(page);
  }

  async _v4Swap(page) {
    const containerSel = "[class*='geetest_box'][style*='block'] [class*='geetest_container']";
    const containerEl = await waitForElement(page, `>CSS> ${containerSel}`, 10000);
    if (!containerEl) throw new Error('GeeTest V4 Swap: container not found');

    const b64 = await screenshotElement(containerEl);
    const containerBox = await containerEl.boundingBox();

    const items = await page.$$("[class*='geetest_item_wrap']");
    const textInstructions = items.length >= 5 ? 'five' : 'three';

    const rawResult = await this.client.click({
      type: 'base64', click: 'geetest', body: b64, textinstructions: textInstructions,
    }, this.debug);
    this._log('GeeTest V4 Swap raw result:', rawResult);
    if (!rawResult || rawResult.includes('ERROR'))
      throw new Error(`GeeTest V4 Swap: bad API result: ${rawResult}`);

    const coordStr = rawResult.includes('coordinates:') ? rawResult.split('coordinates:')[1] : rawResult;
    const pairs = coordStr.split(';').map(p => p.trim()).filter(Boolean);

    for (const pair of pairs) {
      let xVal = 0, yVal = 0;
      for (const part of pair.split(',')) {
        const t = part.trim();
        if (t.startsWith('x=')) xVal = parseInt(t.slice(2).replace(/[^0-9]/g, ''));
        else if (t.startsWith('y=')) yVal = parseInt(t.slice(2).replace(/[^0-9]/g, ''));
      }
      await humanClick(page, containerBox.x + xVal, containerBox.y + yVal);
      await sleep(rand(300, 700));
    }

    await sleep(rand(800, 1500));
    return await this._checkSuccess(page);
  }

  async _dragPrecise(page, fromX, fromY, toX, toY) {
    const dist = toX - fromX;
    if (Math.abs(dist) < 2) return;

    await page.mouse.move(fromX + rand(-2, 2), fromY + rand(-2, 2), { steps: rand(3, 6) });
    await sleep(rand(200, 500));
    await page.mouse.down();
    await sleep(rand(100, 250));

    const overshoot = rand(5, 15);
    const totalDist = dist + overshoot;
    const mainSteps = rand(30, 55);
    let yDrift = 0;
    const yDriftBias = (Math.random() - 0.5) * 0.6;

    for (let i = 1; i <= mainSteps; i++) {
      const p = i / mainSteps;
      let eased;
      if (p < 0.15) {
        eased = p * p * 22.2;
      } else if (p < 0.85) {
        eased = 0.05 + (p - 0.15) / 0.7 * 0.85;
      } else {
        const t = (p - 0.85) / 0.15;
        eased = 0.9 + t * t * 0.1;
      }

      const curX = fromX + totalDist * eased + (Math.random() - 0.5) * 2.5;
      yDrift += (Math.random() - 0.5 + yDriftBias) * 0.7;
      yDrift *= 0.92;
      const curY = fromY + yDrift + (Math.random() - 0.5) * 1.8;

      await page.mouse.move(curX, curY);

      let baseDelay;
      if (p < 0.15) baseDelay = rand(18, 40);
      else if (p < 0.85) baseDelay = rand(8, 18);
      else baseDelay = rand(25, 55);

      if (Math.random() < 0.04) baseDelay += rand(40, 120);
      await sleep(baseDelay);
    }

    await sleep(rand(30, 80));

    const corrSteps = rand(8, 16);
    const currX = fromX + totalDist;
    const corrDist = toX - currX;
    for (let i = 1; i <= corrSteps; i++) {
      const p = i / corrSteps;
      const cx = currX + corrDist * p + (Math.random() - 0.5) * 0.8;
      yDrift *= 0.85;
      const cy = fromY + yDrift + (Math.random() - 0.5) * 0.6;
      await page.mouse.move(cx, cy);
      await sleep(rand(20, 50));
    }

    await page.mouse.move(toX + (Math.random() - 0.5) * 0.5, fromY + (Math.random() - 0.5) * 0.5);
    await sleep(rand(80, 250));
    await page.mouse.up();
  }

  async _checkSuccess(page) {
    return await page.evaluate(() => {
      function isVis(el) {
        if (!el) return false;
        const s = window.getComputedStyle(el);
        return s.display !== 'none' && s.visibility !== 'hidden' && el.offsetWidth > 0;
      }
      const successSels = [
        "div[class*='geetest_lock_success'] [class*='geetest_holder']",
        "div[class*='geetest_radar_success']",
        '.geetest_success_radar_tip_content',
        "[class*='geetest_success_animate']",
        "[class*='geetest_success_radar_tip']",
        "[class*='geetest_success']",
      ];
      for (const sel of successSels) {
        const el = document.querySelector(sel);
        if (el && isVis(el)) return true;
      }
      return false;
    });
  }

  async _checkSolvedAfterDrag(page) {
    if (await this._checkSuccess(page)) return true;
    const windowGone = await page.evaluate(() => {
      function isVis(el) {
        if (!el) return false;
        const s = window.getComputedStyle(el);
        return s.display !== 'none' && s.visibility !== 'hidden' && el.offsetWidth > 0;
      }
      const sels = [
        "[class*='geetest_box'][style*='block']",
        "[class*='geetest_silver'][style*='block']",
        "[class*='geetest_window']",
        "[class*='geetest_holder'][class*='geetest_ant']",
        "[class*='geetest_slider_button']",
      ];
      for (const sel of sels) {
        const el = document.querySelector(sel);
        if (el && isVis(el)) return false;
      }
      return true;
    });
    return windowGone;
  }
}

module.exports = { GeetestSolver };
