const { CaptchaSolverBase } = require('./base_solver');
const { sleep, rand, waitForElement, screenshotElement, humanClick, randomMouseWiggle } = require('./helpers');

const SEL_BUTTON = '#home_children_button, #wrong_children_button, #wrongTimeout_children_button, button[data-theme*="verifyButton"], [class*="game-fail"] .button, .error .button';
const SEL_TITLE = '#game_children_text h2, .tile-game h2, .match-game h2';
const SEL_IMAGE = '#game_challengeItem_image, .challenge-container button, .key-frame-image';

const SEL_ALL_GAME_ELEMENTS = [
  '#game_challengeItem',
  'div[class*="match-game box screen"]',
  'div[class*="tile-game box screen"]',
  'legend[id*="game-header"]',
  'a[id*="Frame_children_arrow"]',
];

const SEL_WRONG_SOLVE = [
  '#wrong_children_exclamation',
  'div[class*="match-game-fail"]',
  'div[class*="tile-game-fail"]',
  'div[id*="descriptionTryAgain"]',
];

const SEL_LOAD_ERROR = [
  'div[class*="error box screen"]',
  '#sub-frame-error',
  '#timeout_widget',
];

const GAME_BOX_TASKS = [
  'coordinatesmatch', 'orbit_match_game', 'seat_coordinates',
  'train_coordinates', 'iconrace', '3d_rollball_objects',
];

class FuncaptchaSolver extends CaptchaSolverBase {
  async solve(selector = '') {
    this._log('Solving FunCaptcha...');
    await randomMouseWiggle(this.page);
    for (let attempt = 0; attempt < this.attempts; attempt++) {
      this._log(`Attempt ${attempt + 1}/${this.attempts}`);
      try {
        if (await this._attempt(selector || this.selector)) {
          this._log('FunCaptcha solved!');
          return true;
        }
      } catch (e) {
        this._log('Error:', e);
        if (attempt < this.attempts - 1) await sleep(rand(1500, 3000));
      }
    }
    throw new Error('FunCaptcha: failed. Error: ERROR_CAPTCHA_UNSOLVABLE');
  }

  async _attempt(sel) {
    const page = this.page;
    const funcaptchaTaskData = {};

    const captureFuncaptcha = async (response) => {
      try {
        const url = response.url();
        if (url.includes('/fc/gfc/') || url.includes('/fc/ca/') || url.includes('funcaptcha')) {
          const ct = response.headers()['content-type'] || '';
          if (ct.includes('json') || ct.includes('text')) {
            const body = await response.text();
            if (body && body.includes('game_data')) {
              try {
                const data = JSON.parse(body);
                if (data && data.game_data) {
                  const gd = data.game_data;
                  if (gd.game_variant) {
                    funcaptchaTaskData.task = gd.game_variant;
                    funcaptchaTaskData.waves = gd.waves || 1;
                  } else if (gd.instruction_string) {
                    funcaptchaTaskData.task = gd.instruction_string;
                    funcaptchaTaskData.waves = gd.waves || 1;
                  }
                }
              } catch {}
            }
          }
        }
      } catch {}
    };

    page.on('response', captureFuncaptcha);
    try {
      return await this._eStart(page, funcaptchaTaskData);
    } finally {
      page.off('response', captureFuncaptcha);
    }
  }

  async _eStart(page, funcaptchaTaskData, n = 25) {
    await this._clickCaptchaTrigger(page);
    const frame = await this._getFuncFrame(page);
    await this._waitForTaskLoad(funcaptchaTaskData);
    await this._clickVerifyButton(page, frame);
    await this._sendButton(frame, 2);

    let funcaptchaType = await this._detectFuncaptchaType(frame, funcaptchaTaskData);
    if (!funcaptchaType) {
      const ver = await this._detectCaptcha(frame);
      if (ver === null) {
        if (await this._checkGreenTick(frame)) {
          this._log('Captcha already solved (green tick)');
          return true;
        }
        this._log('detectCaptcha: type not found');
        return false;
      }
      funcaptchaType = { 0: 'Game_Item', 1: 'Game_Tile', 2: 'Game_Box' }[ver] || 'Game_Box';
    }

    this._log(`FunCaptcha type: ${funcaptchaType}`);

    const task = await this._getTaskTest(frame, funcaptchaTaskData);
    if (!task) { this._log('getTaskTest: task not found'); return false; }
    this._log(`Task: ${task}`);

    const clickMethod = 'funcap';
    this._log(`API click method: ${clickMethod}`);

    let imgOld = '';
    let firstLoad = true;

    for (let i = 0; i < n; i++) {
      this._log(`Solve iteration ${i + 1}/${n}`);

      if (!firstLoad) {
        if (await this._checkWrongSolve(frame)) { this._log('Solved incorrectly'); return false; }
      }

      if (!firstLoad && i > 0) {
        if (await this._checkGreenTick(frame)) { this._log('GREEN TICK — solved!'); return true; }
      }

      if (await this._checkLoadErrors(frame)) {
        this._log('Load error detected');
        const errorBtn = await frame.$('div[class*="error box screen"] button');
        if (errorBtn) { try { await errorBtn.click({ delay: rand(30, 80) }); await sleep(1500); } catch {} }
        continue;
      }

      const btnCheck = await frame.$(SEL_BUTTON);
      if (btnCheck) {
        this._log('Button reappeared, clicking...');
        await this._sendButton(frame, 1);
        await sleep(1000);
        imgOld = '';
      }

      const img = await this._getImageForType(frame, funcaptchaType);

      if (img && imgOld && imgOld === img) {
        this._log('Same image. Checking...');
        await sleep(1000);
        if (await this._checkGreenTick(frame)) { this._log('GREEN TICK after same image — solved!'); return true; }
        const checkBtn = await frame.$(SEL_BUTTON);
        if (!checkBtn) { this._log('No button — solved'); return true; }
        continue;
      }

      if (!img) {
        this._log('No image found, checking...');
        await sleep(1000);
        if (await this._checkGreenTick(frame)) { this._log('GREEN TICK (no image) — solved!'); return true; }
        if (!(await frame.$(SEL_IMAGE))) { this._log('selImage gone — solved'); return true; }
        this._log('selImage present but no image data — Error 5');
        return false;
      }

      firstLoad = false;

      const apiParams = { type: 'base64', click: clickMethod, textinstructions: task, body: img };
      const result = await this.client.click(apiParams, this.debug);

      if (result && String(result).trim() !== '' && String(result).trim() !== 'WAIT' && !String(result).toUpperCase().includes('ERROR')) {
        const resStr = String(result).trim();
        this._log(`API result: ${resStr}`);

        if (resStr.includes('coordinates')) {
          const xMatch = resStr.match(/x=(\d+)/);
          const yMatch = resStr.match(/y=(\d+)/);
          if (xMatch && yMatch) {
            const x = parseInt(xMatch[1]);
            const y = parseInt(yMatch[1]);
            this._log(`Parsed coordinates: x=${x}, y=${y}`);

            if (['Game_Box', 'Game_Children'].includes(funcaptchaType)) {
              const imgWidth = 300;
              const num = Math.floor(x / imgWidth);
              this._log(`Coordinates → image index: ${num}`);

              let arrowSel, submitSel;
              if (funcaptchaType === 'Game_Children') {
                arrowSel = 'a[id*="children_arrowRight"]';
                submitSel = '#game_children_buttonContainer button';
              } else {
                arrowSel = '.right-arrow';
                submitSel = '.button';
              }

              const arrow = await frame.$(arrowSel);
              const btnSubmit = await frame.$(submitSel);
              if (arrow && btnSubmit) {
                for (let j = 0; j < num; j++) { await arrow.click(); await sleep(150); }
                await btnSubmit.click({ delay: rand(30, 80) });
                this._log(`Clicked arrow ${num} times, then submit`);
              } else {
                this._log('Arrow or submit not found');
              }
            } else {
              const col = Math.floor(x / 100);
              const row = Math.floor(y / 100);
              const cellIndex = row * 3 + col;
              this._log(`Coordinates → cell index: ${cellIndex}`);

              let cells;
              if (funcaptchaType === 'Game_Item') cells = await frame.$$('#game_children_challenge a');
              else if (funcaptchaType === 'Game_Tile') cells = await frame.$$('.challenge-container button');
              else if (funcaptchaType === 'Game_Lite') cells = await frame.$$('fieldset > div input, fieldset > div label');
              else cells = await frame.$$('.challenge-container button');

              if (cellIndex >= 0 && cellIndex < cells.length) {
                await cells[cellIndex].click({ delay: rand(30, 80) });
                this._log(`Clicked cell ${cellIndex}`);
              }
            }
          }
        } else {
          const num = this._parseIndex(resStr);
          if (num !== null) {
            if (funcaptchaType === 'Game_Item') {
              const cells = await frame.$$('#game_children_challenge a');
              if (num >= 0 && num < cells.length) { await cells[num].click({ delay: rand(30, 80) }); this._log(`Clicked cell index: ${num}`); }
            } else if (funcaptchaType === 'Game_Tile') {
              const cells = await frame.$$('.challenge-container button');
              if (num >= 0 && num < cells.length) { await cells[num].click({ delay: rand(30, 80) }); this._log(`Clicked cell index: ${num}`); }
            } else if (['Game_Box', 'Game_Children'].includes(funcaptchaType)) {
              let arrowSel, submitSel;
              if (funcaptchaType === 'Game_Children') {
                arrowSel = 'a[id*="children_arrowRight"]';
                submitSel = '#game_children_buttonContainer button';
              } else {
                arrowSel = '.right-arrow';
                submitSel = '.button';
              }
              const arrow = await frame.$(arrowSel);
              const btnSubmit = await frame.$(submitSel);
              if (arrow && btnSubmit) {
                for (let j = 0; j < num; j++) { await arrow.click(); await sleep(150); }
                await btnSubmit.click({ delay: rand(30, 80) });
                this._log(`Clicked arrow ${num} times, then submit`);
              }
            } else if (funcaptchaType === 'Game_Lite') {
              const cells = await frame.$$('fieldset > div input, fieldset > div label');
              if (num >= 0 && num < cells.length) { await cells[num].click({ delay: rand(30, 80) }); this._log(`Clicked cell index: ${num}`); }
            }
          }
        }
      } else {
        this._log(`API Error or WAIT: ${result}`);
      }

      imgOld = img;
      await sleep(1500);

      if (await this._checkGreenTick(frame)) { this._log('GREEN TICK after solve — solved!'); return true; }

      const hasImage = await frame.$(SEL_IMAGE);
      const hasTitle = await frame.$(SEL_TITLE);
      if (!hasImage && !hasTitle) { this._log('Elements gone — solved'); return true; }
    }

    await sleep(1000);
    this._log('Loop exhausted');
    return true;
  }

  async _detectFuncaptchaType(frame, funcaptchaTaskData) {
    const task = funcaptchaTaskData.task || '';
    if (task) {
      for (const boxTask of GAME_BOX_TASKS) {
        if (task.includes(boxTask)) return 'Game_Box';
      }
    }
    const typeSelectors = [
      ['#game_challengeItem', 'Game_Item'],
      ['div[class*="match-game box screen"]', 'Game_Box'],
      ['div[class*="tile-game box screen"]', 'Game_Tile'],
      ['legend[id*="game-header"]', 'Game_Lite'],
      ['a[id*="Frame_children_arrow"]', 'Game_Children'],
    ];
    for (let attempt = 0; attempt < 40; attempt++) {
      for (const [sel, ftype] of typeSelectors) {
        try {
          const el = await frame.$(sel);
          if (el) {
            const visible = await el.evaluate(e => {
              const s = window.getComputedStyle(e);
              return s.display !== 'none' && s.visibility !== 'hidden' && e.offsetWidth > 0;
            });
            if (visible) return ftype;
          }
        } catch {}
      }
      await sleep(200);
    }
    return null;
  }

  async _getImageForType(frame, funcaptchaType) {
    if (funcaptchaType === 'Game_Item') return await this._getImageSrc(frame);

    if (funcaptchaType === 'Game_Box' || funcaptchaType === 'Game_Tile') {
      const minSize = funcaptchaType === 'Game_Box' ? 50000 : 5000;
      for (let attempt = 0; attempt < 30; attempt++) {
        const imgEl = await frame.$('.challenge-container button, .key-frame-image');
        if (imgEl) {
          const b64 = await this._extractImageToCanvas(frame, imgEl);
          if (b64 && b64.length > minSize) {
            this._log(`Got image (size: ${b64.length} chars, attempt ${attempt + 1})`);
            return b64;
          } else if (b64) {
            this._log(`Image too small (${b64.length} chars, need >${minSize}), waiting...`);
          }
        }
        await sleep(500);
      }
      const imgEl = await frame.$('.challenge-container button, .key-frame-image');
      if (imgEl) {
        const b64 = await this._extractImageToCanvas(frame, imgEl);
        if (b64) { this._log(`Fallback image (size: ${b64.length} chars)`); return b64; }
      }
      return null;
    }

    if (funcaptchaType === 'Game_Children') {
      for (let i = 0; i < 20; i++) {
        const imgEl = await frame.$('img[id*="children_challengeImage"]');
        if (imgEl) {
          const b64 = await this._extractImageToCanvas(frame, imgEl);
          if (b64) return b64;
        }
        await sleep(300);
      }
      return null;
    }

    if (funcaptchaType === 'Game_Lite') {
      for (let i = 0; i < 20; i++) {
        const b64 = await frame.evaluate(() => {
          const el = document.querySelector('img[id*="challenge-image"]');
          if (!el) return null;
          if (el.src && el.src.startsWith('data:image'))
            return el.src.replace(/data:image\/[a-z]+;base64,/g, '');
          return null;
        });
        if (b64) return b64;
        await sleep(300);
      }
      return null;
    }

    return await this._getImageSrc(frame);
  }

  async _clickCaptchaTrigger(page) {
    for (let attempt = 0; attempt < 15; attempt++) {
      for (const sel of ['iframe[src*="arkoselabs"]', 'iframe[src*="funcaptcha"]', 'iframe[data-src*="octocaptcha"]']) {
        if (await page.$(sel)) return true;
      }
      if (this.selector) {
        try {
          const trigger = await page.$(this.selector);
          if (trigger) {
            const visible = await trigger.evaluate(e => {
              const s = window.getComputedStyle(e);
              return s.display !== 'none' && s.visibility !== 'hidden' && e.offsetWidth > 0;
            });
            if (visible) {
              this._log(`Clicking trigger: ${this.selector}`);
              const box = await trigger.boundingBox();
              if (box) await humanClick(page, box.x + box.width / 2, box.y + box.height / 2);
              await sleep(rand(1500, 2500));
              return true;
            }
          }
        } catch {}
      }
      await sleep(rand(500, 800));
    }
    return false;
  }

  async _waitForTaskLoad(funcaptchaTaskData, maxWaitMs = 15000) {
    let elapsed = 0;
    while (elapsed < maxWaitMs) {
      if (funcaptchaTaskData.task) {
        this._log(`Task loaded from network after ${elapsed}ms`);
        return true;
      }
      await sleep(200);
      elapsed += 200;
    }
    this._log(`Task not from network after ${maxWaitMs}ms`);
    return false;
  }

  async _clickVerifyButton(page, frame) {
    for (const sel of ['#verifyButton', 'button[data-theme*="verifyButton"]']) {
      try {
        const btn = await frame.$(sel);
        if (btn) {
          const visible = await btn.evaluate(e => {
            const s = window.getComputedStyle(e);
            return s.display !== 'none' && s.visibility !== 'hidden' && e.offsetWidth > 0;
          });
          if (visible) {
            this._log(`Clicking ${sel}`);
            await btn.click({ delay: rand(30, 80) });
            await sleep(2000);
            return;
          }
        }
      } catch {}
    }
    for (const iframeSel of ['iframe[src*="enforcementFrame"]', 'iframe[src*="arkoselabs"]']) {
      try {
        const el = await page.$(iframeSel);
        if (el) {
          const ef = await el.contentFrame();
          if (ef) {
            for (const sel of ['#verifyButton', 'button[data-theme*="verifyButton"]']) {
              const btn = await ef.$(sel);
              if (btn) {
                const visible = await btn.evaluate(e => {
                  const s = window.getComputedStyle(e);
                  return s.display !== 'none' && s.visibility !== 'hidden' && e.offsetWidth > 0;
                });
                if (visible) {
                  await btn.click({ delay: rand(30, 80) });
                  await sleep(2000);
                  return;
                }
              }
            }
          }
        }
      } catch {}
    }
  }

  async _checkGreenTick(frame) {
    for (const sel of SEL_ALL_GAME_ELEMENTS) {
      try {
        const el = await frame.$(sel);
        if (el) {
          const visible = await el.evaluate(e => {
            const s = window.getComputedStyle(e);
            return s.display !== 'none' && s.visibility !== 'hidden' && e.offsetWidth > 0;
          });
          if (visible) return false;
        }
      } catch { return false; }
    }
    return true;
  }

  async _checkWrongSolve(frame) {
    for (const sel of SEL_WRONG_SOLVE) {
      try {
        const el = await frame.$(sel);
        if (el) {
          const visible = await el.evaluate(e => {
            const s = window.getComputedStyle(e);
            return s.display !== 'none' && s.visibility !== 'hidden' && e.offsetWidth > 0;
          });
          if (visible) return true;
        }
      } catch {}
    }
    return false;
  }

  async _checkLoadErrors(frame) {
    for (const sel of SEL_LOAD_ERROR) {
      try {
        const el = await frame.$(sel);
        if (el) {
          const visible = await el.evaluate(e => {
            const s = window.getComputedStyle(e);
            return s.display !== 'none' && s.visibility !== 'hidden' && e.offsetWidth > 0;
          });
          if (visible) return true;
        }
      } catch {}
    }
    return false;
  }

  async _getFuncFrame(page) {
    for (let attempt = 0; attempt < 20; attempt++) {
      for (const iframeSel of [
        'iframe[src*="arkoselabs"]', 'iframe[src*="funcaptcha"]',
        'iframe[id*="enforcementFrame"]', 'iframe[data-src*="octocaptcha"]',
      ]) {
        try {
          const el = await page.$(iframeSel);
          if (el) {
            const f1 = await el.contentFrame();
            if (f1) {
              const inner = await f1.$('iframe');
              if (inner) {
                const f2 = await inner.contentFrame();
                if (f2) {
                  const gc = await f2.$('#game-core-frame, iframe[src*="game-core"]');
                  if (gc) {
                    const f3 = await gc.contentFrame();
                    if (f3) { this._log('Found game-core-frame (3 levels)'); return f3; }
                  }
                  if (await f2.$('#game_challengeItem, .tile-game, .match-game, legend[id*="game-header"]')) {
                    this._log('Found game frame (2 levels)');
                    return f2;
                  }
                }
              }
              const gc = await f1.$('#game-core-frame, iframe[src*="game-core"]');
              if (gc) {
                const f2 = await gc.contentFrame();
                if (f2) { this._log('Found game-core-frame (2 levels)'); return f2; }
              }
              if (await f1.$('#game_challengeItem, .tile-game, .match-game, legend[id*="game-header"]')) {
                this._log('Found game frame (1 level)');
                return f1;
              }
            }
          }
        } catch {}
      }
      if (attempt < 19) await sleep(500);
    }
    for (const iframeSel of ['iframe[src*="arkoselabs"]', 'iframe[src*="funcaptcha"]']) {
      const el = await page.$(iframeSel);
      if (el) {
        const f = await el.contentFrame();
        if (f) return f;
      }
    }
    return page;
  }

  async _sendButton(frame, maxTries = 5) {
    if (!(await frame.$(SEL_BUTTON))) return true;
    for (let i = 0; i < maxTries; i++) {
      const btn = await frame.$(SEL_BUTTON);
      if (!btn) return true;
      try {
        const visible = await btn.evaluate(e => {
          const s = window.getComputedStyle(e);
          return s.display !== 'none' && s.visibility !== 'hidden' && e.offsetWidth > 0;
        });
        if (!visible) return true;
        await btn.click({ delay: rand(30, 80) });
        this._log('Clicked start/retry button');
        await sleep(1000);
        if (!(await frame.$(SEL_BUTTON))) return true;
      } catch (e) {
        this._log('Button click error:', e);
      }
      await sleep(1000);
    }
    return false;
  }

  async _detectCaptcha(frame) {
    for (let i = 0; i < 20; i++) {
      for (const [sel, val] of [['script[src*="tile-game-ui"]', 0], ['.tile-game', 1], ['.match-game', 2]]) {
        if (await frame.$(sel)) return val;
      }
      await sleep(500);
    }
    return null;
  }

  async _getTaskTest(frame, funcaptchaTaskData) {
    const task = funcaptchaTaskData.task || '';
    if (task) return task;
    for (let i = 0; i < 20; i++) {
      try {
        const el = await frame.$(SEL_TITLE);
        if (el) {
          const text = await el.evaluate(e => (e.innerText || '').replace(/\s+/g, ' ').trim());
          if (text) return text;
        }
      } catch {}
      await sleep(200);
    }
    return funcaptchaTaskData.task || '';
  }

  async _getImageSrc(frame) {
    for (let i = 0; i < 20; i++) {
      const b64 = await frame.evaluate(() => {
        const el = document.querySelector('#game_challengeItem_image, .challenge-container button, .key-frame-image');
        if (!el) return null;
        if ('src' in el && el.src != null && el.src !== '')
          return el.src.replace(/data:image\/[a-z]+;base64,/g, '');
        return null;
      });
      if (b64) return b64;
      await sleep(500);
    }
    return null;
  }

  async _extractImageToCanvas(frame, imgEl) {
    const b64 = await frame.evaluate(async (el) => {
      function tryCanvasRender(imgObj, w, h) {
        try {
          if (!w || !h) return null;
          let c = document.createElement('canvas');
          c.width = w; c.height = h;
          let ctx = c.getContext('2d');
          ctx.drawImage(imgObj, 0, 0);
          try { let d = ctx.getImageData(0, 0, 1, 1).data; if (d[3] === 0) return null; } catch { return null; }
          return c.toDataURL('image/jpeg').replace(/data:image\/[a-z]+;base64,/g, '');
        } catch { return null; }
      }

      async function fetchAsBase64(url) {
        try {
          let resp = await fetch(url);
          let blob = await resp.blob();
          return await new Promise((resolve) => {
            let reader = new FileReader();
            reader.onloadend = () => {
              let dataUrl = reader.result;
              resolve(dataUrl ? dataUrl.replace(/data:[^;]+;base64,/g, '') : null);
            };
            reader.onerror = () => resolve(null);
            reader.readAsDataURL(blob);
          });
        } catch { return null; }
      }

      let bg = getComputedStyle(el).backgroundImage;
      if (!bg || bg === 'none') {
        if ('src' in el && el.src) bg = 'url("' + el.src + '")';
        else return null;
      }

      if ('computedStyleMap' in el && !/url\(["']https?:\/\//.test(bg)) {
        try {
          let styleImg = el.computedStyleMap().get('background-image');
          if (styleImg instanceof CSSImageValue) {
            let w = el.clientWidth, h = el.clientHeight;
            if (w && h) {
              let c = document.createElement('canvas');
              c.width = w; c.height = h;
              c.getContext('2d').drawImage(styleImg, 0, 0);
              try {
                let d = c.getContext('2d').getImageData(0, 0, 1, 1).data;
                if (d[3] !== 0) {
                  let r = c.toDataURL('image/jpeg').replace(/data:image\/[a-z]+;base64,/g, '');
                  if (r) return r;
                }
              } catch {}
            }
          }
        } catch {}
      }

      let urlMatch = /"(.+)"/.exec(bg);
      if (!urlMatch) return null;
      let imageUrl = urlMatch[1];

      if (imageUrl.startsWith('data:image'))
        return imageUrl.replace(/data:[^;]+;base64,/g, '');

      try {
        let a = document.createElement('a'); a.href = imageUrl;
        if (new URL(a.href).origin === document.location.origin) {
          let img = new Image(); img.crossOrigin = 'anonymous'; img.src = imageUrl;
          if (img.complete && img.naturalWidth > 0) {
            let r = tryCanvasRender(img, img.naturalWidth, img.naturalHeight);
            if (r) return r;
          }
        }
      } catch {}

      let fetched = await fetchAsBase64(imageUrl);
      if (fetched) return fetched;

      return null;
    }, imgEl);

    if (b64) return b64;

    try { return await screenshotElement(imgEl); } catch { return null; }
  }

  _parseIndex(resultStr) {
    try {
      const match = String(resultStr).match(/\d+/);
      if (match) return parseInt(match[0]) - 1;
    } catch {}
    return null;
  }
}

module.exports = { FuncaptchaSolver };
