const https = require('https');
const http = require('http');
const { sleep } = require('./helpers');

function request(url, options = {}) {
  return new Promise((resolve, reject) => {
    const parsed = new URL(url);
    const mod = parsed.protocol === 'https:' ? https : http;
    const body = options.body || null;
    const reqOpts = {
      hostname: parsed.hostname,
      port: parsed.port || (parsed.protocol === 'https:' ? 443 : 80),
      path: parsed.pathname + parsed.search,
      method: options.method || 'GET',
      headers: { ...(options.headers || {}) },
      timeout: options.timeout || 30000,
    };
    if (body) {
      const buf = Buffer.from(body, 'utf8');
      reqOpts.headers['Content-Length'] = buf.length;
    }
    const req = mod.request(reqOpts, (res) => {
      const chunks = [];
      res.on('data', (c) => chunks.push(c));
      res.on('end', () => {
        const text = Buffer.concat(chunks).toString('utf8');
        resolve({ status: res.statusCode, text });
      });
    });
    req.on('error', reject);
    req.on('timeout', () => { req.destroy(); reject(new Error('timeout')); });
    if (body) req.write(body);
    req.end();
  });
}

class CapGuruClient {
  static SERVERS = [
    'https://api2.cap.guru',
    'https://api.cap.guru',
    'https://api5.cap.guru',
    'https://api4.cap.guru',
    'https://api3.cap.guru',
    'https://ipv6.cap.guru',
  ];

  constructor(apiKey, serverUrl = 'https://api2.cap.guru') {
    this.apiKey = apiKey;
    this.serverUrl = serverUrl.replace(/\/+$/, '');
    this._workingServer = this.serverUrl;
    this._lastTaskId = null;
  }

  async _post(payload, debug = false) {
    payload.key = this.apiKey;
    payload.softguru = '154076';

    const body = JSON.stringify(payload);
    const servers = [this.serverUrl, ...CapGuruClient.SERVERS.filter(s => s !== this.serverUrl)];
    let lastErr = null;

    for (const server of servers) {
      try {
        const resp = await request(`${server}/in.php`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Accept': 'application/json, text/plain, */*',
          },
          body,
          timeout: 30000,
        });
        if (resp.status < 200 || resp.status >= 300) throw new Error(`HTTP ${resp.status}`);
        const text = resp.text.trim();
        if (debug) console.log(`[CapGuruClient] in.php (${server}):`, text.slice(0, 300));
        this._workingServer = server;
        return text;
      } catch (e) {
        lastErr = e;
        if (debug) console.log(`[CapGuruClient] server ${server} failed:`, e.message);
      }
    }
    throw new Error(`Cap.Guru: all servers failed. Last error: ${lastErr}`);
  }

  async _getResult(taskId, debug = false) {
    const server = this._workingServer;
    try {
      const params = new URLSearchParams({ action: 'get', key: this.apiKey, id: taskId });
      const resp = await request(`${server}/res.php?${params}`, { timeout: 15000 });
      if (resp.status < 200 || resp.status >= 300) throw new Error(`HTTP ${resp.status}`);
      const text = resp.text.trim();
      if (debug) console.log(`[CapGuruClient] res.php (${server}) id=${taskId}:`, text.slice(0, 300));
      return text;
    } catch (e) {
      if (debug) console.log(`[CapGuruClient] res.php (${server}) failed:`, e.message);
      return await this._getResultFallback(taskId, debug);
    }
  }

  async _getResultFallback(taskId, debug = false) {
    let lastErr = null;
    for (const server of CapGuruClient.SERVERS) {
      try {
        const params = new URLSearchParams({ action: 'get', key: this.apiKey, id: taskId });
        const resp = await request(`${server}/res.php?${params}`, { timeout: 15000 });
        if (resp.status < 200 || resp.status >= 300) throw new Error(`HTTP ${resp.status}`);
        const text = resp.text.trim();
        if (debug) console.log(`[CapGuruClient] res.php fallback (${server}) id=${taskId}:`, text.slice(0, 300));
        this._workingServer = server;
        return text;
      } catch (e) {
        lastErr = e;
        if (debug) console.log(`[CapGuruClient] res.php fallback (${server}) failed:`, e.message);
      }
    }
    throw new Error(`Cap.Guru res.php: all servers failed. Last error: ${lastErr}`);
  }

  async send(payload, debug = false) {
    const text = await this._post(payload, debug);
    if (text.startsWith('OK|')) {
      const taskId = text.split('|')[1];
      this._lastTaskId = taskId;
      return taskId;
    }
    if (text.includes('NOT')) return 'WAIT';
    throw new Error(`Cap.Guru create task error: ${text}`);
  }

  async get(taskId, debug = false) {
    const text = await this._getResult(taskId, debug);
    if (text.startsWith('OK|')) return text.split('|')[1];
    if (text.includes('NOT') || text === 'CAPCHA_NOT_READY') return 'WAIT';
    throw new Error(`Cap.Guru result error: ${text}`);
  }

  async click(payload, debug = false) {
    const taskIdStr = await this.send(payload, debug);
    if (taskIdStr.includes('ERROR') || taskIdStr.includes('WAIT')) return null;
    const taskId = taskIdStr.trim();
    await sleep(5000);
    for (let i = 0; i < 60; i++) {
      const result = await this.get(taskId, debug);
      if (result === 'WAIT') {
        await sleep(3000);
        continue;
      }
      if (result.includes('ERROR') && !result.includes('notpic')) return null;
      return result;
    }
    return null;
  }

  async solveRecap(imageB64, taskText, sizex = 9, debug = false) {
    return await this.click({
      type: 'base64', click: 'recap2', textinstructions: taskText,
      sizex, bas_disable_image_convert: '1', body: imageB64,
    }, debug);
  }

  async solveCoordinatesRaw({
    click: clickVal,
    body0 = null,
    body1 = null,
    body = null,
    textinstructions = null,
    vernet = null,
    coordinatescaptcha = '1',
    bas_disable_image_convert = null,
    debug = false,
  } = {}) {
    const payload = { type: 'base64', click: clickVal };
    if (body0 !== null) payload.body0 = body0;
    if (body1 !== null) payload.body1 = body1;
    if (body !== null) payload.body = body;
    payload.coordinatescaptcha = coordinatescaptcha;
    if (textinstructions) payload.textinstructions = textinstructions;
    if (vernet !== null) payload.vernet = String(vernet);
    if (bas_disable_image_convert) payload.bas_disable_image_convert = bas_disable_image_convert;
    return await this.click(payload, debug);
  }

  parseTileIndices(raw) {
    const indices = [];
    const cleaned = raw.replace(/[^0-9,]/g, '');
    for (const part of cleaned.split(',')) {
      const trimmed = part.trim();
      if (trimmed) {
        const n = parseInt(trimmed, 10);
        if (!isNaN(n)) indices.push(n - 1);
      }
    }
    return indices;
  }

  parseRawPixelCoords(raw) {
    return (raw.match(/[0-9]+/g) || []).map(Number);
  }
}

module.exports = { CapGuruClient };
