const { CapGuruClient } = require('./capguru_client');
const { sleep, rand } = require('./helpers');

class CaptchaSolverBase {
  constructor({ page, apiKey, server = 'https://api.cap.guru', attempts = 5, debug = false, selector = '' }) {
    this.page = page;
    this.apiKey = apiKey;
    this.server = server;
    this.attempts = attempts;
    this.debug = debug;
    this.selector = selector;
    this.client = new CapGuruClient(apiKey, server);
  }

  _log(...args) {
    if (this.debug) console.log('[CaptchaSolver]', ...args);
  }

  async _urlToBase64(url) {
    try {
      const result = await this.page.evaluate(async (imgUrl) => {
        const response = await fetch(imgUrl);
        if (!response.ok) return null;
        const blob = await response.blob();
        return new Promise((resolve) => {
          const reader = new FileReader();
          reader.onloadend = () => resolve(reader.result.split(',')[1]);
          reader.readAsDataURL(blob);
        });
      }, url);
      if (result) return result;
    } catch {}
    return await this._serverFetchB64(url);
  }

  async _serverFetchB64(url) {
    try {
      const client = await this.page.createCDPSession();
      try {
        const { body, base64Encoded } = (await client.send('Network.loadNetworkResource', {
          frameId: (await client.send('Page.getFrameTree')).frameTree.frame.id,
          url: url,
          options: { disableCache: false, includeCredentials: true },
        })).resource;
        if (body) return base64Encoded ? body : Buffer.from(body).toString('base64');
      } finally {
        await client.detach().catch(() => {});
      }
    } catch {}

    try {
      const cookies = await this.page.cookies(url);
      const cookieHeader = cookies.map(c => `${c.name}=${c.value}`).join('; ');
      const mod = url.startsWith('https') ? require('https') : require('http');
      return await new Promise((resolve) => {
        const req = mod.get(url, { headers: { 'Cookie': cookieHeader } }, (res) => {
          if (res.statusCode >= 300 && res.statusCode < 400 && res.headers.location) {
            const redir = res.headers.location.startsWith('http') ? res.headers.location : new URL(res.headers.location, url).href;
            const mod2 = redir.startsWith('https') ? require('https') : require('http');
            mod2.get(redir, { headers: { 'Cookie': cookieHeader } }, (res2) => {
              if (res2.statusCode < 200 || res2.statusCode >= 300) { resolve(null); return; }
              const chunks = [];
              res2.on('data', (c) => chunks.push(c));
              res2.on('end', () => { const buf = Buffer.concat(chunks); resolve(buf.length > 100 ? buf.toString('base64') : null); });
              res2.on('error', () => resolve(null));
            }).on('error', () => resolve(null));
            return;
          }
          if (res.statusCode < 200 || res.statusCode >= 300) { resolve(null); return; }
          const chunks = [];
          res.on('data', (c) => chunks.push(c));
          res.on('end', () => { const buf = Buffer.concat(chunks); resolve(buf.length > 100 ? buf.toString('base64') : null); });
          res.on('error', () => resolve(null));
        });
        req.on('error', () => resolve(null));
        req.setTimeout(15000, () => { req.destroy(); resolve(null); });
      });
    } catch { return null; }
  }

  async _dragSlider(page, fromX, fromY, toX, toY) {
    await page.mouse.move(fromX, fromY);
    await sleep(rand(80, 180));
    await page.mouse.down();
    await sleep(rand(100, 300));

    const steps = rand(25, 45);
    const dist = toX - fromX;
    for (let i = 1; i <= steps; i++) {
      const p = i / steps;
      const ease = p < 0.5 ? 2 * p * p : -1 + (4 - 2 * p) * p;
      const curX = fromX + dist * ease + (Math.random() * 3 - 1.5);
      const curY = fromY + (Math.random() * 4 - 2);
      await page.mouse.move(curX, curY);
      await sleep(rand(8, 22));
    }

    await page.mouse.move(toX, fromY);
    await sleep(rand(80, 200));
    await page.mouse.up();
  }
}

module.exports = { CaptchaSolverBase };
