let capUse = false;
let capUseCount = 0;
let onoffk1 = false, onoffk2 = false;

// Скрытие автоматизации
try {
    Object.defineProperty(navigator, 'webdriver', {
        get: () => undefined,
        configurable: true
    });
} catch(e) {}

try {
    delete window.chrome;
    delete window.navigator.chrome;
} catch(e) {}

// Улучшенные профили с индивидуальными особенностями
const HUMAN_PROFILES = [
    // Осторожный, точный (пожилой/внимательный)
    { 
        mouseSpeed: 380, reactionTime: [320, 950], accuracy: 0.97, 
        hesitation: 0.38, typingSpeed: [95, 240], handSteadiness: 0.92,
        errorRate: 0.02, readingSpeed: 'slow', impulsivity: 0.15
    },
    // Быстрый, небрежный (торопится)
    { 
        mouseSpeed: 1250, reactionTime: [80, 320], accuracy: 0.78, 
        hesitation: 0.08, typingSpeed: [25, 85], handSteadiness: 0.65,
        errorRate: 0.08, readingSpeed: 'fast', impulsivity: 0.75
    },
    // Средний, сбалансированный
    { 
        mouseSpeed: 720, reactionTime: [180, 520], accuracy: 0.90, 
        hesitation: 0.22, typingSpeed: [50, 140], handSteadiness: 0.85,
        errorRate: 0.04, readingSpeed: 'normal', impulsivity: 0.45
    },
    // Медленный, методичный
    { 
        mouseSpeed: 290, reactionTime: [480, 1400], accuracy: 0.98, 
        hesitation: 0.52, typingSpeed: [140, 350], handSteadiness: 0.95,
        errorRate: 0.01, readingSpeed: 'very_slow', impulsivity: 0.10
    },
    // Опытный, но иногда торопится
    { 
        mouseSpeed: 980, reactionTime: [120, 380], accuracy: 0.87, 
        hesitation: 0.15, typingSpeed: [35, 110], handSteadiness: 0.80,
        errorRate: 0.05, readingSpeed: 'fast', impulsivity: 0.60
    }
];

const CURRENT_PROFILE = HUMAN_PROFILES[Math.floor(secureRandom() * HUMAN_PROFILES.length)];
const _hour = new Date().getHours();
const TIME_FACTOR = (_hour >= 23 || _hour < 6) ? 1.45
                  : (_hour >= 6  && _hour < 9)  ? 1.20
                  : (_hour >= 13 && _hour < 15)  ? 1.12
                  : (_hour >= 17 && _hour < 19)  ? 1.08
                  : 1.0;

let sessionFatigue = 0;
let _actionCount = 0;
let currentFocus = null; // Текущий элемент в фокусе внимания

function getCurrentProfile() {
    _actionCount++;
    sessionFatigue = Math.min(sessionFatigue + 0.12, 50);
    const fatigueFactor = Math.min(sessionFatigue / 100, 0.40) * TIME_FACTOR;
    const timeOfDayModifier = (_hour >= 14 && _hour <= 16) ? 1.1 : 1.0; // Послеобеденная сонливость
    
    return {
        mouseSpeed: CURRENT_PROFILE.mouseSpeed * (1 - fatigueFactor * 0.50) / TIME_FACTOR,
        reactionTime: [
            CURRENT_PROFILE.reactionTime[0] * (1 + fatigueFactor) * TIME_FACTOR * timeOfDayModifier,
            CURRENT_PROFILE.reactionTime[1] * (1 + fatigueFactor * 0.7) * TIME_FACTOR * timeOfDayModifier
        ],
        accuracy: Math.max(0.68, CURRENT_PROFILE.accuracy * (1 - fatigueFactor * 0.22)),
        hesitation: CURRENT_PROFILE.hesitation * (1 + fatigueFactor * 1.2),
        typingSpeed: [
            CURRENT_PROFILE.typingSpeed[0] * (1 + fatigueFactor * 0.6) * TIME_FACTOR,
            CURRENT_PROFILE.typingSpeed[1] * (1 + fatigueFactor * 1.0) * TIME_FACTOR
        ],
        handSteadiness: Math.max(0.5, CURRENT_PROFILE.handSteadiness * (1 - fatigueFactor * 0.3)),
        errorRate: Math.min(0.15, CURRENT_PROFILE.errorRate * (1 + fatigueFactor * 0.5)),
        impulsivity: Math.max(0.05, CURRENT_PROFILE.impulsivity * (1 - fatigueFactor * 0.2))
    };
}

// Улучшенный генератор случайных чисел
function secureRandom() {
    if (window.crypto && window.crypto.getRandomValues) {
        const arr = new Uint32Array(2);
        window.crypto.getRandomValues(arr);
        return (arr[0] * 4294967296 + arr[1]) / 18446744073709551616;
    }
    return Math.random();
}

// Гауссово распределение с улучшенной точностью
function randomGaussian(mean = 0, std = 1) {
    let u = 0, v = 0;
    while (u === 0) u = secureRandom();
    while (v === 0) v = secureRandom();
    const z = Math.sqrt(-2.0 * Math.log(u)) * Math.cos(2.0 * Math.PI * v);
    return z * std + mean;
}

function randomLogNormal(mean, sigma) {
    const normal = randomGaussian(0, 1);
    return Math.exp(mean + sigma * normal);
}

// Улучшенная функция задержек с эмоциональными состояниями
async function humanDelay(type = 'reaction') {
    const profile = getCurrentProfile();
    let delay;
    
    switch(type) {
        case 'reaction':
            delay = randomLogNormal(5.6, 0.42);
            delay = Math.max(profile.reactionTime[0], Math.min(profile.reactionTime[1], delay));
            break;
        case 'decision':
            // Бимодальное: быстрое решение vs долгое обдумывание
            if (secureRandom() < profile.impulsivity) {
                delay = randomGaussian(280, 75);
            } else {
                delay = randomGaussian(1200 + sessionFatigue * 12, 350);
            }
            // Иногда "залипание" на элементе
            if (secureRandom() < 0.08) delay += randomGaussian(800, 200);
            break;
        case 'typing':
            const base = randomLogNormal(4.25, 0.32);
            delay = Math.max(profile.typingSpeed[0], Math.min(profile.typingSpeed[1], base));
            // Пауза на размышление при наборе
            if (secureRandom() < 0.03) delay += randomGaussian(400, 120);
            break;
        case 'reading':
            delay = randomLogNormal(6.8, 0.55);
            delay = Math.max(550, Math.min(2800, delay));
            // Регрессия взгляда (перечитывание)
            if (secureRandom() < 0.15) delay *= 1.4;
            break;
        case 'micro':
            delay = Math.abs(randomGaussian(40, 15));
            break;
        case 'pause':
            delay = secureRandom() < 0.18
                ? randomGaussian(2800, 750)
                : randomGaussian(520, 160);
            break;
        case 'contemplate':
            delay = randomLogNormal(7.5, 0.62);
            delay = Math.max(700, Math.min(4500, delay));
            // Долгое изучение сложного элемента
            if (secureRandom() < 0.12) delay *= 1.6;
            break;
        case 'correction':
            // Время на исправление ошибки
            delay = randomGaussian(450, 120);
            break;
        case 'hesitation':
            // Нерешительность
            delay = randomGaussian(600, 180);
            break;
        default:
            delay = randomGaussian(300, 85);
    }
    
    const jitter = randomGaussian(0, delay * 0.09);
    const finalDelay = Math.max(12, Math.floor(delay + jitter));
    
    // Редкий "лаг" системы
    if (secureRandom() < 0.035) {
        await new Promise(resolve => setTimeout(resolve, finalDelay + Math.floor(secureRandom() * 220)));
    } else {
        await new Promise(resolve => setTimeout(resolve, finalDelay));
    }
    
    sessionFatigue = Math.max(0, sessionFatigue - 0.05);
}

// Улучшенное управление указателем
let lastPointerId = Math.floor(secureRandom() * 10000);
function getNextPointerId() {
    lastPointerId = (lastPointerId + Math.floor(secureRandom() * 4) + 1) % 100000;
    return lastPointerId;
}

// История движений с предсказанием
const mouseHistory = {
    x: window.innerWidth / 2,
    y: window.innerHeight / 2,
    timestamp: Date.now(),
    velocity: { x: 0, y: 0 },
    path: [],
    attentionLevel: 1.0
};

function updateMouseHistory(x, y) {
    const now = Date.now();
    const timeDelta = Math.max(1, now - mouseHistory.timestamp);
    
    const newVelocity = {
        x: (x - mouseHistory.x) / timeDelta * 16,
        y: (y - mouseHistory.y) / timeDelta * 16
    };
    
    mouseHistory.path.push({
        x: mouseHistory.x,
        y: mouseHistory.y,
        time: mouseHistory.timestamp,
        velocity: { ...mouseHistory.velocity }
    });
    
    if (mouseHistory.path.length > 15) mouseHistory.path.shift();
    
    // Плавное изменение скорости
    mouseHistory.velocity.x = mouseHistory.velocity.x * 0.7 + newVelocity.x * 0.3;
    mouseHistory.velocity.y = mouseHistory.velocity.y * 0.7 + newVelocity.y * 0.3;
    
    mouseHistory.x = x;
    mouseHistory.y = y;
    mouseHistory.timestamp = now;
    
    return timeDelta;
}

function getScreenProps() {
    const dpr = window.devicePixelRatio || 1;
    const jitterX = randomGaussian(0, 0.4);
    const jitterY = randomGaussian(0, 0.4);
    return {
        screenX: Math.floor(mouseHistory.x * dpr + jitterX),
        screenY: Math.floor(mouseHistory.y * dpr + jitterY),
        clientX: mouseHistory.x,
        clientY: mouseHistory.y,
        movementX: mouseHistory.path.length > 1 
            ? mouseHistory.x - mouseHistory.path[mouseHistory.path.length - 2].x 
            : 0,
        movementY: mouseHistory.path.length > 1 
            ? mouseHistory.y - mouseHistory.path[mouseHistory.path.length - 2].y 
            : 0
    };
}

function buildYandexTaskUrl(url) {
    try {
        if (!url || typeof url !== 'string') return '';
        if (/([?&])(?:b?data)=task\b/i.test(url)) return url;
        if (/([?&])(?:b?data)=img\b/i.test(url)) {
            return url.replace(/([?&])((?:b)?data)=img\b/i, '$1$2=task');
        }
        const sep = url.includes('?') ? '&' : '?';
        return url + sep + 'data=task';
    } catch (e) {
        return '';
    }
}

// Улучшенный движок движений с физикой
class HumanMovement {
    constructor() {
        this.velocity = { x: 0, y: 0 };
        this.acceleration = { x: 0, y: 0 };
        this.lastPauseTime = 0;
    }
    
    calculateTrajectory(start, end, duration) {
        const distance = Math.hypot(end.x - start.x, end.y - start.y);
        const profile = getCurrentProfile();
        
        // Закон Фиттса с индивидуальными коэффициентами
        const fittsID = Math.log2(distance / 10 + 1);
        const fittsTime = duration * fittsID * 0.15;
        const actualDuration = fittsTime * (1 + randomGaussian(0, 0.15));
        
        // Адаптивная плотность точек
        const baseFrameMs = Math.max(6, 14 - (profile.mouseSpeed - 300) / 100);
        const steps = Math.max(20, Math.floor(actualDuration / baseFrameMs));
        
        // Случайная кривизна с асимметрией для правшей/левшей
        const isRightHanded = secureRandom() < 0.88; // 88% правшей
        const handBias = isRightHanded ? 1 : -1;
        const curvature = randomGaussian(distance * 0.03 * handBias, distance * 0.12);
        const perpAngle = Math.atan2(end.y - start.y, end.x - start.x) + Math.PI / 2;
        
        // Возможные паттерны движения
        const patternType = secureRandom();
        const hasWiggle = patternType < 0.15;
        const hasPause = patternType < 0.35 && distance > 200;
        const hasOvershoot = secureRandom() > profile.accuracy && distance > 100;
        
        const cp1Ratio = 0.22 + secureRandom() * 0.15;
        const cp2Ratio = 0.65 + secureRandom() * 0.15;
        
        const controlPoint1 = {
            x: start.x + (end.x - start.x) * cp1Ratio + Math.cos(perpAngle) * curvature,
            y: start.y + (end.y - start.y) * cp1Ratio + Math.sin(perpAngle) * curvature
        };
        const controlPoint2 = {
            x: start.x + (end.x - start.x) * cp2Ratio + Math.cos(perpAngle) * curvature * 0.25,
            y: start.y + (end.y - start.y) * cp2Ratio + Math.sin(perpAngle) * curvature * 0.25
        };
        
        const points = [];
        const now = Date.now();
        let pauseIndex = hasPause ? Math.floor(steps * (0.4 + secureRandom() * 0.3)) : -1;
        let pauseDuration = hasPause ? randomGaussian(180, 50) : 0;
        
        for (let i = 0; i <= steps; i++) {
            let t = i / steps;
            let jerkT = this.minimumJerk(t);
            
            // Добавление паузы в середине
            if (i === pauseIndex) {
                jerkT = this.minimumJerk(t);
            }
            
            let pos = this.cubicBezier(jerkT, start, controlPoint1, controlPoint2, end);
            
            // Зигзаг в середине пути
            if (hasWiggle) {
                const wiggleCenter = 0.5;
                const wiggleWidth = 0.08;
                const wt = Math.exp(-Math.pow((t - wiggleCenter) / wiggleWidth, 2));
                const wiggleAmp = randomGaussian(0, distance * 0.05);
                pos.x += Math.cos(perpAngle) * wiggleAmp * wt;
                pos.y += Math.sin(perpAngle) * wiggleAmp * wt;
            }
            
            // Физиологический тремор
            const tremor = this.calculateTremor(t, steps, profile.handSteadiness);
            
            // Шум уменьшается к цели
            const noiseScale = 0.5 * (1 - jerkT * 0.6);
            const noise = {
                x: randomGaussian(0, noiseScale),
                y: randomGaussian(0, noiseScale)
            };
            
            // Микро-остановки
            const microStop = (secureRandom() < 0.012) ? randomGaussian(30, 12) : 0;
            
            let timestamp = now + (actualDuration * t) + microStop;
            if (i >= pauseIndex && pauseIndex > 0) {
                timestamp += pauseDuration;
            }
            
            points.push({
                x: pos.x + tremor.x + noise.x,
                y: pos.y + tremor.y + noise.y,
                t: t,
                timestamp: timestamp,
                isPause: i === pauseIndex
            });
        }
        
        return points;
    }
    
    minimumJerk(t) {
        return 10 * Math.pow(t, 3) - 15 * Math.pow(t, 4) + 6 * Math.pow(t, 5);
    }
    
    cubicBezier(t, p0, p1, p2, p3) {
        const cX = 3 * (p1.x - p0.x);
        const bX = 3 * (p2.x - p1.x) - cX;
        const aX = p3.x - p0.x - cX - bX;
        const cY = 3 * (p1.y - p0.y);
        const bY = 3 * (p2.y - p1.y) - cY;
        const aY = p3.y - p0.y - cY - bY;
        
        return {
            x: (aX * Math.pow(t, 3)) + (bX * Math.pow(t, 2)) + (cX * t) + p0.x,
            y: (aY * Math.pow(t, 3)) + (bY * Math.pow(t, 2)) + (cY * t) + p0.y
        };
    }
    
    calculateTremor(t, totalSteps, steadiness) {
        // Физиологический тремор: 8-12 Гц
        const freq1 = 8.2 + randomGaussian(0, 0.9);
        const freq2 = 11.5 + randomGaussian(0, 0.7);
        const envelope = Math.sin(t * Math.PI); // Максимум в середине
        const amplitude = (0.25 + envelope * 0.50) * (2 - steadiness) * (1 + sessionFatigue * 0.015);
        
        const phase1 = t * Math.PI * 2 * freq1;
        const phase2 = t * Math.PI * 2 * freq2 + secureRandom() * Math.PI;
        
        return {
            x: (Math.cos(phase1) * 0.6 + Math.cos(phase2) * 0.4) * amplitude * randomGaussian(1, 0.12),
            y: (Math.sin(phase1) * 0.6 + Math.sin(phase2) * 0.4) * amplitude * randomGaussian(1, 0.12)
        };
    }
}

const movementEngine = new HumanMovement();

// Улучшенное движение к элементу
async function humanMoveTo(element, targetX, targetY, options = {}) {
    const rect = element.getBoundingClientRect();
    const startX = options.startX || mouseHistory.x;
    const startY = options.startY || mouseHistory.y;
    
    const target = {
        x: rect.left + targetX,
        y: rect.top + targetY
    };
    
    const profile = getCurrentProfile();
    const distance = Math.hypot(target.x - startX, target.y - startY);
    const baseDuration = (distance / profile.mouseSpeed) * 1000;
    
    // Решение о пролёте мимо цели
    const willOvershoot = secureRandom() > profile.accuracy && !options.noOvershoot && distance > 80;
    let actualTarget = { ...target };
    let overshootReturn = false;
    
    if (willOvershoot) {
        const overshootDist = randomLogNormal(3.2, 0.6);
        const angle = Math.atan2(target.y - startY, target.x - startX);
        actualTarget.x += Math.cos(angle) * overshootDist;
        actualTarget.y += Math.sin(angle) * overshootDist;
        overshootReturn = true;
    }
    
    // Иногда движение проходит мимо с возвратом (коррекция траектории)
    const willPassBy = secureRandom() < 0.12 && distance > 150;
    if (willPassBy && !willOvershoot) {
        const passAngle = Math.atan2(target.y - startY, target.x - startX) + (secureRandom() < 0.5 ? 0.3 : -0.3);
        const passDist = distance * (0.6 + secureRandom() * 0.2);
        actualTarget.x = startX + Math.cos(passAngle) * passDist;
        actualTarget.y = startY + Math.sin(passAngle) * passDist;
        overshootReturn = true;
    }
    
    const points = movementEngine.calculateTrajectory(
        { x: startX, y: startY },
        actualTarget,
        baseDuration
    );
    
    for (let i = 0; i < points.length; i++) {
        const point = points[i];
        updateMouseHistory(point.x, point.y);
        const screenProps = getScreenProps();
        
        const event = new MouseEvent('mousemove', {
            bubbles: true,
            cancelable: true,
            view: window,
            detail: 0,
            screenX: screenProps.screenX,
            screenY: screenProps.screenY,
            clientX: point.x,
            clientY: point.y,
            ctrlKey: false,
            altKey: false,
            shiftKey: false,
            metaKey: false,
            button: 0,
            buttons: 0,
            relatedTarget: element,
            movementX: screenProps.movementX,
            movementY: screenProps.movementY
        });
        
        Object.defineProperty(event, 'timeStamp', { value: point.timestamp, writable: false });
        element.dispatchEvent(event);
        
        // Нерешительность в середине пути
        if (secureRandom() < profile.hesitation * 0.6 && i > points.length * 0.4 && i < points.length * 0.7) {
            await humanDelay('micro');
        }
        
        if (i < points.length - 1) {
            const nextPoint = points[i + 1];
            let waitTime = nextPoint.timestamp - point.timestamp;
            
            // Замедление к цели
            const progress = i / points.length;
            if (progress > 0.7) waitTime *= 1.2;
            
            const jitteredWait = waitTime * (1 + randomGaussian(0, 0.12));
            await new Promise(r => setTimeout(r, Math.max(0, jitteredWait)));
        }
    }
    
    // Возврат при промахе
    if (overshootReturn && !options.noOvershoot) {
        await humanDelay('micro');
        const returnDuration = randomGaussian(180, 60) * (willPassBy ? 1.5 : 1);
        const returnPoints = movementEngine.calculateTrajectory(actualTarget, target, returnDuration);
        
        for (const point of returnPoints) {
            updateMouseHistory(point.x, point.y);
            const screenProps = getScreenProps();
            
            element.dispatchEvent(new MouseEvent('mousemove', {
                bubbles: true,
                cancelable: true,
                view: window,
                screenX: screenProps.screenX,
                screenY: screenProps.screenY,
                clientX: point.x,
                clientY: point.y,
                movementX: screenProps.movementX,
                movementY: screenProps.movementY
            }));
            
            await new Promise(r => setTimeout(r, 14));
        }
    }
    
    return { x: target.x, y: target.y };
}

// Улучшенный клик с недокликами и двойными кликами
async function humanClick(element, options = {}) {
    if (typeof element === 'string') element = document.querySelector(element);
    if (!element) return false;
    
    const rect = element.getBoundingClientRect();
    const profile = getCurrentProfile();
    
    // Выбор точки клика с учётом размера элемента
    const marginX = Math.min(rect.width * 0.12, 15);
    const marginY = Math.min(rect.height * 0.12, 15);
    const targetX = marginX + secureRandom() * (rect.width - marginX * 2);
    const targetY = marginY + secureRandom() * (rect.height - marginY * 2);
    
    await humanDelay('decision');
    
    // Иногда мышь проходит мимо, потом возвращается
    if (secureRandom() < 0.15) {
        const passX = targetX + randomGaussian(rect.width * 0.5, 20);
        const passY = targetY + randomGaussian(0, 15);
        await humanMoveTo(element, passX, passY, options);
        await humanDelay('micro');
    }
    
    await humanMoveTo(element, targetX, targetY, options);
    await humanDelay('reaction');
    
    // Микрокоррекция позиции
    if (secureRandom() < 0.65) {
        const adjustX = targetX + randomGaussian(0, 1.5);
        const adjustY = targetY + randomGaussian(0, 1.5);
        await humanMoveTo(element, adjustX, adjustY, { noOvershoot: true });
        await humanDelay('micro');
    }
    
    const preClickX = mouseHistory.x + randomGaussian(0, 0.4);
    const preClickY = mouseHistory.y + randomGaussian(0, 0.4);
    
    // Вариативность задержки hover -> mousedown
    const hoverPause = secureRandom() < 0.35
        ? randomGaussian(420, 100)
        : randomGaussian(85, 25);
    
    // Длительность нажатия с вариативностью
    const pressDuration = Math.max(50, randomLogNormal(4.7, 0.42));
    
    // Решение о типе клика
    const clickTypeRoll = secureRandom();
    const isDoubleClick = clickTypeRoll < 0.02; // 2% двойных кликов
    const isMissedClick = clickTypeRoll < 0.05 && clickTypeRoll >= 0.02; // 3% промахов
    
    const events = [
        { type: 'mouseover', time: 0 },
        { type: 'mouseenter', time: randomGaussian(6, 2) },
        { type: 'mousemove', time: randomGaussian(14, 4) },
        { type: 'mousedown', time: hoverPause },
        { type: 'mouseup', time: pressDuration },
        { type: 'click', time: randomGaussian(10, 3) }
    ];
    
    // Выполнение первого клика
    for (const evt of events) {
        await new Promise(r => setTimeout(r, Math.max(0, evt.time)));
        
        const screenProps = getScreenProps();
        const pointerId = getNextPointerId();
        
        if (['mousedown', 'mouseup'].includes(evt.type) || secureRandom() > 0.25) {
            const pointerType = evt.type === 'mousedown' ? 'pointerdown' : 
                               evt.type === 'mouseup' ? 'pointerup' : 'pointermove';
            
            element.dispatchEvent(new PointerEvent(pointerType, {
                bubbles: true,
                cancelable: true,
                view: window,
                detail: evt.type === 'click' ? 1 : 0,
                screenX: screenProps.screenX,
                screenY: screenProps.screenY,
                clientX: preClickX,
                clientY: preClickY,
                ctrlKey: false,
                altKey: false,
                shiftKey: false,
                metaKey: false,
                button: evt.type === 'mousedown' ? 0 : -1,
                buttons: evt.type === 'mousedown' ? 1 : 0,
                relatedTarget: null,
                pointerId: pointerId,
                width: 1 + secureRandom() * 0.8,
                height: 1 + secureRandom() * 0.8,
                pressure: evt.type === 'mousedown' ? 0.5 + secureRandom() * 0.35 : 0,
                tiltX: randomGaussian(0, 6),
                tiltY: randomGaussian(0, 6),
                pointerType: 'mouse',
                isPrimary: true
            }));
        }
        
        element.dispatchEvent(new MouseEvent(evt.type, {
            bubbles: true,
            cancelable: true,
            view: window,
            detail: evt.type === 'click' ? 1 : 0,
            screenX: screenProps.screenX,
            screenY: screenProps.screenY,
            clientX: preClickX,
            clientY: preClickY,
            ctrlKey: false,
            altKey: false,
            shiftKey: false,
            metaKey: false,
            button: evt.type === 'mousedown' ? 0 : -1,
            buttons: evt.type === 'mousedown' ? 1 : 0,
            relatedTarget: null,
            movementX: 0,
            movementY: 0
        }));
        
        if (evt.type === 'mousedown' && (element.tagName === 'INPUT' || element.tagName === 'TEXTAREA')) {
            element.focus();
            element.dispatchEvent(new FocusEvent('focus', { bubbles: true }));
        }
    }
    
    // Двойной клик
    if (isDoubleClick) {
        await humanDelay('micro');
        const dblClickEvents = ['mousedown', 'mouseup', 'click', 'dblclick'];
        for (const evtType of dblClickEvents) {
            element.dispatchEvent(new MouseEvent(evtType, {
                bubbles: true,
                cancelable: true,
                clientX: preClickX,
                clientY: preClickY
            }));
            await new Promise(r => setTimeout(r, randomGaussian(80, 20)));
        }
    }
    
    // "Недоклик" - быстрое отпускание без полного нажатия
    if (isMissedClick && !isDoubleClick) {
        await humanDelay('micro');
        element.dispatchEvent(new MouseEvent('mousedown', {
            bubbles: true,
            cancelable: true,
            clientX: preClickX,
            clientY: preClickY
        }));
        await new Promise(r => setTimeout(r, 30));
        element.dispatchEvent(new MouseEvent('mouseup', {
            bubbles: true,
            cancelable: true,
            clientX: preClickX,
            clientY: preClickY
        }));
        // Повторный клик
        await humanDelay('micro');
        element.dispatchEvent(new MouseEvent('click', {
            bubbles: true,
            cancelable: true,
            clientX: preClickX,
            clientY: preClickY
        }));
    }
    
    // Движение после клика (не все сразу убирают мышь)
    if (secureRandom() < 0.45) {
        await humanDelay('micro');
        const moveAwayX = targetX + randomGaussian(25, 12);
        const moveAwayY = targetY + randomGaussian(18, 10);
        await humanMoveTo(element, moveAwayX, moveAwayY, { noOvershoot: true });
    }
    
    return true;
}

// Улучшенное перетаскивание
async function humanDrag(element, distance, options = {}) {
    if (typeof element === 'string') element = document.querySelector(element);
    if (!element) return false;
    
    const rect = element.getBoundingClientRect();
    const startX = rect.left + rect.width / 2;
    const startY = rect.top + rect.height / 2;
    
    await humanMoveTo(element, 0, 0, { startX: mouseHistory.x, startY: mouseHistory.y });
    await humanDelay('reaction');
    
    const grabX = mouseHistory.x;
    const grabY = mouseHistory.y;
    
    // Захват
    element.dispatchEvent(new PointerEvent('pointerdown', {
        bubbles: true,
        cancelable: true,
        pointerId: getNextPointerId(),
        pointerType: 'mouse',
        clientX: grabX,
        clientY: grabY,
        pressure: 0.8,
        buttons: 1
    }));
    
    element.dispatchEvent(new MouseEvent('mousedown', {
        bubbles: true,
        cancelable: true,
        clientX: grabX,
        clientY: grabY,
        button: 0,
        buttons: 1
    }));
    
    await humanDelay('micro');
    
    const profile = getCurrentProfile();
    const steps = Math.floor(distance / 2.5) + Math.floor(randomGaussian(25, 8));
    const targetX = startX + distance;
    
    const willOverdrag = secureRandom() < 0.25;
    const actualTargetX = willOverdrag ? targetX + randomGaussian(25, 15) : targetX;
    
    let yDrift = 0;
    const yDriftDir = secureRandom() < 0.5 ? 1 : -1;
    let lastPause = 0;
    
    for (let i = 0; i <= steps; i++) {
        const progress = i / steps;
        
        // Ease in-out с случайными отклонениями
        const easedProgress = progress < 0.5 
            ? 4 * progress * progress * progress 
            : 1 - Math.pow(-2 * progress + 2, 3) / 2;
        
        let currentX = startX + (actualTargetX - startX) * easedProgress;
        
        // Возврат при перетаскивании
        if (progress > 0.85 && willOverdrag) {
            currentX = targetX + (actualTargetX - targetX) * (1 - (progress - 0.85) / 0.15);
        }
        
        // Y: плавный дрейф + тремор
        yDrift += randomGaussian(0, 0.4) * yDriftDir;
        yDrift *= 0.90;
        let currentY = startY + yDrift + randomGaussian(0, 1.5);
        
        updateMouseHistory(currentX, currentY);
        const screenProps = getScreenProps();
        
        element.dispatchEvent(new MouseEvent('mousemove', {
            bubbles: true,
            cancelable: true,
            clientX: currentX,
            clientY: currentY,
            screenX: screenProps.screenX,
            screenY: screenProps.screenY,
            movementX: screenProps.movementX,
            movementY: screenProps.movementY,
            buttons: 1
        }));
        
        // Скорость с ускорением и замедлением
        const speedCurve = Math.sin(progress * Math.PI);
        const baseWait = 16 / (0.8 + speedCurve * 0.4);
        const extraWait = secureRandom() < 0.05 ? randomGaussian(45, 15) : 0;
        
        // Редкие паузы (рука колеблется)
        if (secureRandom() < 0.025 && i - lastPause > 20) {
            await humanDelay('micro');
            lastPause = i;
        }
        
        await new Promise(r => setTimeout(r, Math.max(6, baseWait + randomGaussian(0, 3) + extraWait)));
    }
    
    await humanDelay('micro');
    
    // Отпускание
    element.dispatchEvent(new PointerEvent('pointerup', {
        bubbles: true,
        cancelable: true,
        pointerId: getNextPointerId(),
        pointerType: 'mouse',
        clientX: mouseHistory.x,
        clientY: mouseHistory.y,
        pressure: 0
    }));
    
    element.dispatchEvent(new MouseEvent('mouseup', {
        bubbles: true,
        cancelable: true,
        clientX: mouseHistory.x,
        clientY: mouseHistory.y,
        button: 0
    }));
    
    return true;
}

// Улучшенный ввод текста с реалистичными ошибками
async function humanType(element, text) {
    if (typeof element === 'string') element = document.querySelector(element);
    if (!element) return;
    
    await humanClick(element);
    await humanDelay('decision');
    
    // Очистка существующего текста
    if (element.value && element.value.length > 0) {
        const clearMethod = secureRandom() < 0.7 ? 'selectall' : 'backspace';
        
        if (clearMethod === 'selectall') {
            element.dispatchEvent(new KeyboardEvent('keydown', {
                key: 'a', code: 'KeyA', ctrlKey: true, bubbles: true
            }));
            element.setSelectionRange(0, element.value.length);
            await humanDelay('micro');
            element.dispatchEvent(new KeyboardEvent('keydown', {
                key: 'Backspace', code: 'Backspace', keyCode: 8, bubbles: true
            }));
        } else {
            // Посимвольное удаление (медленнее, но реалистичнее)
            for (let i = 0; i < Math.min(element.value.length, 20); i++) {
                element.dispatchEvent(new KeyboardEvent('keydown', {
                    key: 'Backspace', code: 'Backspace', keyCode: 8, bubbles: true
                }));
                element.value = element.value.slice(0, -1);
                await humanDelay('typing');
            }
        }
        
        element.value = '';
        await humanDelay('micro');
    }
    
    const chars = text.split('');
    let consecutiveCorrect = 0;
    let lastWasError = false;
    
    for (let i = 0; i < chars.length; i++) {
        const char = chars[i];
        const profile = getCurrentProfile();
        
        // Вероятность ошибки растёт с усталостью и скоростью
        const errorChance = lastWasError ? 0.01 : Math.min(0.08, consecutiveCorrect * 0.008 + profile.errorRate);
        
        // Решение об ошибке
        if (secureRandom() < errorChance && char.match(/[a-zA-Zа-яА-Я0-9]/) && !lastWasError) {
            // Генерация похожей ошибки
            const errorTypes = ['adjacent', 'miss', 'double', 'case'];
            const errorType = errorTypes[Math.floor(secureRandom() * errorTypes.length)];
            let wrongChar;
            
            switch(errorType) {
                case 'adjacent':
                    const charCode = char.charCodeAt(0);
                    wrongChar = String.fromCharCode(charCode + (secureRandom() < 0.5 ? 1 : -1));
                    break;
                case 'miss':
                    wrongChar = char; // Пропуск (будет исправлен как backspace)
                    break;
                case 'double':
                    await typeChar(element, char);
                    wrongChar = char;
                    break;
                case 'case':
                    wrongChar = char === char.toUpperCase() ? char.toLowerCase() : char.toUpperCase();
                    break;
            }
            
            if (errorType !== 'miss') {
                await typeChar(element, wrongChar);
                consecutiveCorrect = 0;
                lastWasError = true;
                
                // Время на осознание ошибки
                await humanDelay(secureRandom() < 0.3 ? 'hesitation' : 'correction');
                
                // Исправление
                await typeChar(element, 'Backspace', true);
                await humanDelay('micro');
            }
        }
        
        await typeChar(element, char);
        consecutiveCorrect++;
        lastWasError = false;
        
        // Паузы в зависимости от символа
        if (char === ' ' || char === '.' || char === ',' || char === '!' || char === '?') {
            await humanDelay('reaction');
        } else if (char === char.toUpperCase() && char !== char.toLowerCase()) {
            // Замедление на заглавные (нужно нажимать Shift)
            await humanDelay('typing');
            await humanDelay('micro');
        } else {
            await humanDelay('typing');
        }
        
        // Случайные паузы на размышление
        if (secureRandom() < 0.04) {
            await humanDelay('decision');
        }
        
        // Пауза после длинного слова
        if (i > 0 && chars[i-1] !== ' ' && char === ' ' && secureRandom() < 0.3) {
            await humanDelay('micro');
        }
    }
    
    await humanDelay('reaction');
}

async function typeChar(element, char, isBackspace = false) {
    const actualChar = isBackspace ? '' : char;
    const code = isBackspace ? 'Backspace' : 'Key' + char.toUpperCase();
    const keyCode = isBackspace ? 8 : char.charCodeAt(0);
    const isUpperCase = char !== char.toLowerCase() && char === char.toUpperCase();
    
    if (!isBackspace) {
        const start = element.selectionStart || element.value.length;
        const end = element.selectionEnd || start;
        element.value = element.value.slice(0, start) + char + element.value.slice(end);
        element.selectionStart = element.selectionEnd = start + 1;
    } else {
        element.value = element.value.slice(0, -1);
    }
    
    // keydown с учётом Shift
    const keydown = new KeyboardEvent('keydown', {
        key: char,
        code: code,
        keyCode: keyCode,
        which: keyCode,
        bubbles: true,
        cancelable: true,
        ctrlKey: false,
        altKey: false,
        shiftKey: isUpperCase,
        metaKey: false,
        repeat: false,
        isComposing: false
    });
    element.dispatchEvent(keydown);
    
    if (!isBackspace) {
        const keypress = new KeyboardEvent('keypress', {
            key: char,
            charCode: keyCode,
            keyCode: keyCode,
            which: keyCode,
            bubbles: true,
            cancelable: true
        });
        element.dispatchEvent(keypress);
    }
    
    // input event
    const input = new InputEvent('input', {
        bubbles: true,
        cancelable: true,
        inputType: isBackspace ? 'deleteContentBackward' : 'insertText',
        data: isBackspace ? null : char,
        isComposing: false
    });
    element.dispatchEvent(input);
    
    await new Promise(r => setTimeout(r, randomGaussian(12, 4)));
    
    // keyup
    const keyup = new KeyboardEvent('keyup', {
        key: char,
        code: code,
        keyCode: keyCode,
        which: keyCode,
        bubbles: true,
        cancelable: true,
        ctrlKey: false,
        altKey: false,
        shiftKey: isUpperCase,
        metaKey: false,
        repeat: false
    });
    element.dispatchEvent(keyup);
}

// Улучшенное блуждание мыши с паттернами внимания
async function idleMouseWander() {
    if (capUse) return;
    
    const profile = getCurrentProfile();
    const wanderType = secureRandom();
    
    if (wanderType < 0.4) {
        // Случайное блуждание
        const rx = randomGaussian(mouseHistory.x, 120);
        const ry = randomGaussian(mouseHistory.y, 80);
        const cx = Math.max(10, Math.min(window.innerWidth - 10, rx));
        const cy = Math.max(10, Math.min(window.innerHeight - 10, ry));
        
        updateMouseHistory(cx, cy);
        document.dispatchEvent(new MouseEvent('mousemove', {
            bubbles: true,
            clientX: cx,
            clientY: cy,
            screenX: Math.floor(cx * (window.devicePixelRatio || 1)),
            screenY: Math.floor(cy * (window.devicePixelRatio || 1))
        }));
    } else if (wanderType < 0.7) {
        // Движение к краю экрана (отдых)
        const edge = Math.floor(secureRandom() * 4);
        let tx, ty;
        switch(edge) {
            case 0: tx = 50; ty = secureRandom() * window.innerHeight; break;
            case 1: tx = window.innerWidth - 50; ty = secureRandom() * window.innerHeight; break;
            case 2: tx = secureRandom() * window.innerWidth; ty = 50; break;
            case 3: tx = secureRandom() * window.innerWidth; ty = window.innerHeight - 50; break;
        }
        
        const steps = 15;
        for (let i = 0; i <= steps; i++) {
            const t = i / steps;
            const x = mouseHistory.x + (tx - mouseHistory.x) * t;
            const y = mouseHistory.y + (ty - mouseHistory.y) * t;
            updateMouseHistory(x, y);
            document.dispatchEvent(new MouseEvent('mousemove', {
                bubbles: true,
                clientX: x,
                clientY: y
            }));
            await new Promise(r => setTimeout(r, 20));
        }
    } else {
        // Микродвижения (чтение/осмотр)
        for (let i = 0; i < 5; i++) {
            const dx = randomGaussian(0, 30);
            const dy = randomGaussian(0, 20);
            updateMouseHistory(mouseHistory.x + dx, mouseHistory.y + dy);
            document.dispatchEvent(new MouseEvent('mousemove', {
                bubbles: true,
                clientX: mouseHistory.x,
                clientY: mouseHistory.y
            }));
            await new Promise(r => setTimeout(r, randomGaussian(150, 40)));
        }
    }
}

// Инициализация
document.addEventListener('DOMContentLoaded', async function() {
    mouseHistory.x = randomGaussian(window.innerWidth / 2, 150);
    mouseHistory.y = randomGaussian(window.innerHeight / 2, 120);
    
    // Случайное начальное положение
    setTimeout(() => {
        document.dispatchEvent(new MouseEvent('mousemove', {
            bubbles: true,
            clientX: mouseHistory.x,
            clientY: mouseHistory.y
        }));
    }, randomGaussian(500, 200));
    
    // Блуждание с переменным интервалом
    setInterval(() => {
        if (secureRandom() < 0.6) idleMouseWander();
    }, 2500 + Math.floor(secureRandom() * 6000));
    
    // Случайная прокрутка
    setInterval(() => {
        if (!capUse && secureRandom() < 0.15) {
            const scrollAmount = randomGaussian(0, 300);
            window.scrollBy({ top: scrollAmount, behavior: 'smooth' });
        }
    }, 8000 + Math.floor(secureRandom() * 10000));
    
    let timerId = setInterval(async function() {
        try {
            // Закрытие GDPR
            let gdpr = document.querySelector('#gdpr-popup-v3-button-all');
            if (gdpr && isVisible(gdpr)) {
                await humanDelay('reaction');
                await humanClick(gdpr);
                await new Promise(r => setTimeout(r, 600));
            }
        } catch(e) {}
        
        let onoffk1 = !await onoff('oth');
        if (onoffk1) { 
            if (onoffk2 == onoffk1) return false; 
            log("Plugin OFF");
            onoffk2 = onoffk1;
            return false;
        } else { 
            if (onoffk2 != onoffk1) capUse = false;
            onoffk2 = false; 
        }
        
        let el = document.querySelector(".AdvancedCaptcha_kaleidoscope, .CaptchaSlider, .AdvancedCaptcha, .CheckboxCaptcha-Anchor, .SliderCaptcha-Inner");
        
        if (el && isVisible(el)) {
            if (!capUse) {
                capUse = true;
                await eStart();
            }
        }
    }, 2200 + Math.floor(randomGaussian(0, 300)));
});

async function eStart() {
    let form1 = 0, form2 = 0, form3 = 0, form4 = 0, form5 = 0, form6 = 0;
    
    let el;
    el = document.querySelector(".CheckboxCaptcha-Anchor"); 
    if (el && isVisible(el)) form1 = 1;
    el = document.querySelector(".CaptchaSlider"); 
    if (el && isVisible(el)) form2 = 1;
    el = document.querySelector(".AdvancedCaptcha-ImageWrapper"); 
    if (el && isVisible(el)) form3 = 1;
    el = document.querySelector(".AdvancedCaptcha_kaleidoscope"); 
    if (el && isVisible(el)) form4 = 1;
    el = document.querySelector(".AdvancedCaptcha-View>img"); 
    if (el && isVisible(el)) form5 = 1;
    
    if (form1 == 0 && form2 == 0 && form3 == 0 && form4 == 0 && form5 == 0) { 
        log("Не нашли следов капчи!");   
        return false; 
    }
    
    for (let i = 0; i < 5; i++) { 
        if (!await onoff('oth')) { 
            log("Plugin OFF in cycle");  
            return false; 
        }
        
        log(form1, form2, form3, form4, form5, form6);
        
        // Улучшенное изучение капчи
        await humanDelay('contemplate');
        
        // Дополнительное наблюдение за сложными заданиями
        if (form3 > 0 || form4 > 0) {
            await humanDelay('reading');
        }
        
        if ((form1 > 0 || form2 > 0) && form3 == 0 && form4 == 0 && form5 == 0) {
            if (form1 > 0) {
                el = document.querySelector(".CheckboxCaptcha-Anchor");
                if (el.className.indexOf("pending") === -1) {
                    await new Promise(r => setTimeout(r, 2200));
                    await humanClick(".CheckboxCaptcha-Anchor input");
                } else {
                    log("Complete click");
                    return true;
                }
            } else if (form2 > 0) {
                el = document.querySelector(".CaptchaSlider");
                let box = el.getBoundingClientRect();
                await humanDrag(".CaptchaSlider .Thumb", box.width);
                await new Promise(r => setTimeout(r, 2500));
            }
            form1 = 0;
            form2 = 0;
            form6 = 1;
        }
        
        if (form3 > 0 || form4 > 0 || form5 > 0 || form6 > 0) {
            el = document.querySelector(".AdvancedCaptcha-ImageWrapper"); 
            if (el && isVisible(el)) { form3 = 1; }
            el = document.querySelector(".AdvancedCaptcha_kaleidoscope"); 
            if (el && isVisible(el)) { form4 = 1; }
            el = document.querySelector(".AdvancedCaptcha-View>img"); 
            if (el && isVisible(el)) { form5 = 1; }
            
            let task = "";
            let imgList = [];
            let puzzle = "";
            
            if (form3 > 0) {
                task = "figur";
                el = document.querySelector(".AdvancedCaptcha-ImageWrapper img");
                
                let img4 = await getBase64FromImageUrlYa(el.src, 2);
                let img1 = img4 && img4[0] ? img4[0] : "";
                const finalUrl = img4 && img4[1] ? img4[1] : el.src;
                const url2 = buildYandexTaskUrl(finalUrl);
                
                let img2 = "";
                if (url2) {
                    img2 = await getBase64FromImageUrlYa(url2);
                }
                
                if (!img2) {
                    try {
                        const canvasEl = document.querySelector(".AdvancedCaptcha-View canvas");
                        if (canvasEl) {
                            img2 = canvasEl.toDataURL().split(",")[1];
                        }
                    } catch (e) {
                        img2 = "";
                    }
                }
                
                if (!img1 || !img2) {
                    log('YandexSmartCaptcha: failed to extract images (img/task)');
                    return false;
                }
                
                imgList.push(img1);
                imgList.push(img2);
                
            } else if (form4 > 0) {
                task = "puzzle";
                let bodyS = window.document.body ? window.document.body.innerHTML : "";
                let parsed = null;
                try {
                    if (window.__SSR_DATA__) {
                        parsed = window.__SSR_DATA__;
                    }
                } catch(e) {}
                
                // Изучение canvas
                if (secureRandom() < 0.7) {
                    const canvasEl = document.querySelector('.AdvancedCaptcha-View canvas');
                    if (canvasEl) {
                        await humanMoveTo(canvasEl,
                            secureRandom() * canvasEl.offsetWidth,
                            secureRandom() * canvasEl.offsetHeight
                        );
                        await humanDelay('reading');
                    }
                }
                
                if (parsed && parsed.imageSrc && parsed.task) {
                    puzzle = Array.isArray(parsed.task) ? parsed.task.join(',') : String(parsed.task);
                    imgList = await getBase64FromImageUrlYa(parsed.imageSrc);
                } else {
                    let matTask = bodyS.match(/"task"\s*:\s*\[([^\]]+)\]/i) || bodyS.match(/\[([0-9\,\s]+)\]/i);
                    let matImg = bodyS.match(/"imageSrc"\s*:\s*"(.*?)"/i) || bodyS.match(/\,imageSrc\:\"(.*?)\"/i);
                    if (matTask && matImg) {
                        puzzle = String(matTask[1]).replace(/\s+/g, '').replace(/[^0-9,]/g, '');
                        imgList = await getBase64FromImageUrlYa(matImg[1]);
                    } else {
                        log("Not solved sorry", task);
                        return false;
                    }
                }
            } else if (form5 > 0) {
                task = "text";
                el = document.querySelector(".AdvancedCaptcha-View>img");
                let img1 = await getBase64FromImageUrlYa(el.src);
                imgList = img1;
            }
            
            let res = await sendChrome.X('imgs', {
                'click': "ya", 
                "type": task, 
                'url': imgList, 
                'puzzle': puzzle
            });
            
            console.log("res", res);
            
            if (res && 'received' in res && res['received'] && res['received'] != '' 
                && res['received'] != 'WAIT' && !hasSrt(res['received'], 'ERROR')) {        
                
                let res1 = res['received'];
                
                if (task == "figur") {
                    res1 = res1.replace(/[^0-9,\;]/g, "");
                    let res2 = res1.split(";");
                    
                    for (let k in res2) {
                        let wq = res2[k].split(",");
                        let xp = parseInt(wq[0]), yp = parseInt(wq[1]);
                        
                        await humanDelay('decision');
                        await humanClickOnCanvas(xp, yp);
                    }
                    
                    await new Promise(r => setTimeout(r, 2500));
                    
                    el = document.querySelector("button[data-testid=\"submit\"]");
                    if (el) { 
                        await humanClick(el);
                        await new Promise(r => setTimeout(r, 3500));
                    }
                    
                } else if (task == "text") {
                    el = document.querySelector(".Textinput-Control");
                    await humanType(el, res1);
                    
                    await new Promise(r => setTimeout(r, 2500));
                    simulateEnterPress(".Textinput-Control");
                    log("Click");
                    await new Promise(r => setTimeout(r, 3500));
                    
                } else if (task == "puzzle") {
                    res1 = parseInt(res1);
                    el = document.querySelector(".CaptchaSlider");
                    let bowWidth = el.getBoundingClientRect();
                    let bowWidth2 = bowWidth.width;
                    el = document.querySelector(".CaptchaSlider .Thumb");
                    
                    if (el) {
                        await humanDragToValue(el, res1, bowWidth2);
                        await new Promise(r => setTimeout(r, 1200));
                        let submitEl = document.querySelector("button[data-testid=\"submit\"]");
                        if (submitEl) {
                            await humanClick(submitEl);
                            log("Puzzle submitted");
                            await new Promise(r => setTimeout(r, 3500));
                            return true;
                        } else {
                            log("Submit button not found");
                        }
                    } else {
                        console.log("bad");
                    }
                }
                
            } else {
                log("Can't solved");
                
                el = document.querySelector("button[data-testid=\"refresh\"]");
                if (el) { 
                    await humanClick(el);
                    await new Promise(r => setTimeout(r, 3500));
                } else {
                    log("Не найти кнопку ресет");
                    return;
                }
            }
        }
    }
}

async function humanClickOnCanvas(x, y) {
    let canvas = document.querySelector(".AdvancedCaptcha-ImageWrapper");
    if (!canvas) return;
    
    let rect = canvas.getBoundingClientRect();
    
    await humanMoveTo(canvas, x, y);
    await humanDelay('micro');
    
    const pointerId = getNextPointerId();
    const screenProps = getScreenProps();
    
    const pointerDown = new PointerEvent('pointerdown', {
        bubbles: true,
        cancelable: true,
        pointerId: pointerId,
        pointerType: 'mouse',
        clientX: mouseHistory.x,
        clientY: mouseHistory.y,
        screenX: screenProps.screenX,
        screenY: screenProps.screenY,
        width: 1 + secureRandom() * 0.6,
        height: 1 + secureRandom() * 0.6,
        pressure: 0.5 + secureRandom() * 0.35,
        tiltX: randomGaussian(0, 4),
        tiltY: randomGaussian(0, 4),
        isPrimary: true
    });
    
    canvas.dispatchEvent(pointerDown);
    
    await new Promise(r => setTimeout(r, randomLogNormal(4.3, 0.25)));
    
    const pointerUp = new PointerEvent('pointerup', {
        bubbles: true,
        cancelable: true,
        pointerId: pointerId,
        pointerType: 'mouse',
        clientX: mouseHistory.x,
        clientY: mouseHistory.y,
        pressure: 0
    });
    
    canvas.dispatchEvent(pointerUp);
}

async function humanDragToValue(thumbElement, targetValue, maxWidth) {
    const rect = thumbElement.getBoundingClientRect();
    const startX = rect.left + rect.width / 2;
    const startY = rect.top + rect.height / 2;
    
    await humanMoveTo(thumbElement, 0, 0, { startX: mouseHistory.x, startY: mouseHistory.y });
    await humanDelay('reaction');
    
    const pointerDown = new PointerEvent('pointerdown', {
        bubbles: true,
        cancelable: true,
        pointerId: getNextPointerId(),
        pointerType: 'mouse',
        clientX: mouseHistory.x,
        clientY: mouseHistory.y,
        pressure: 0.8,
        buttons: 1
    });
    thumbElement.dispatchEvent(pointerDown);
    
    const mousedown = new MouseEvent('mousedown', {
        bubbles: true,
        cancelable: true,
        clientX: mouseHistory.x,
        clientY: mouseHistory.y,
        button: 0,
        buttons: 1
    });
    thumbElement.dispatchEvent(mousedown);
    
    await humanDelay('micro');
    
    const sliderEl = thumbElement.closest('.CaptchaSlider');
    const sliderRect = sliderEl ? sliderEl.getBoundingClientRect() : null;
    const trackWidth = (sliderRect && sliderRect.width) ? sliderRect.width : maxWidth;
    const maxDistance = Math.max(0, trackWidth - rect.width);
    
    const ariaMaxRaw = thumbElement.getAttribute('aria-valuemax');
    const ariaMax = ariaMaxRaw ? parseInt(ariaMaxRaw, 10) : 100;
    const ariaMinRaw = thumbElement.getAttribute('aria-valuemin');
    const ariaMin = ariaMinRaw ? parseInt(ariaMinRaw, 10) : 0;
    
    let targetAria = Number.isFinite(targetValue) ? Math.trunc(targetValue) : ariaMax;
    if (targetAria > ariaMax + 2) {
        if (maxDistance > 0) {
            targetAria = Math.round((targetAria / maxDistance) * (ariaMax - ariaMin) + ariaMin);
        } else if (trackWidth > 0) {
            targetAria = Math.round((targetAria / trackWidth) * (ariaMax - ariaMin) + ariaMin);
        }
    }
    targetAria = Math.max(ariaMin, Math.min(ariaMax, targetAria));
    
    const stepPxBase = 4;
    let i = 0;
    let lastAria = null;
    let safety = 0;
    let lastPause = 0;
    
    for (i = 0; i <= maxDistance && safety < 2500; i += stepPxBase) {
        safety++;
        const currentX = startX + i + randomGaussian(0, 0.8);
        const currentY = startY + randomGaussian(0, 1.5);
        
        updateMouseHistory(currentX, currentY);
        const screenProps = getScreenProps();
        
        thumbElement.dispatchEvent(new MouseEvent('mousemove', {
            bubbles: true,
            cancelable: true,
            clientX: currentX,
            clientY: currentY,
            screenX: screenProps.screenX,
            screenY: screenProps.screenY,
            movementX: screenProps.movementX,
            movementY: screenProps.movementY,
            buttons: 1
        }));
        
        // Улучшенная скорость с ускорением и замедлением
        const progressToTarget = (ariaMax > ariaMin && targetAria > ariaMin)
            ? Math.min(1, Math.max(0, ((lastAria || ariaMin) - ariaMin) / (targetAria - ariaMin)))
            : 0;
        const speedMs = 45 + Math.floor((1 - Math.sin(progressToTarget * Math.PI * 0.5)) * 90)
                      + Math.floor(secureRandom() * 55)
                      + (secureRandom() < 0.06 ? Math.floor(randomGaussian(85, 28)) : 0);
        
        await new Promise(r => setTimeout(r, speedMs));
        
        const nowRaw = thumbElement.getAttribute('aria-valuenow');
        const nowAria = nowRaw ? parseInt(nowRaw, 10) : null;
        if (nowAria !== null) {
            lastAria = nowAria;
            
            if (nowAria === targetAria) break;
            if (nowAria > targetAria) {
                let back = 0;
                while (back < 30) {
                    back++;
                    const bx = (startX + i) - back; 
                    updateMouseHistory(bx, currentY);
                    const sp = getScreenProps();
                    thumbElement.dispatchEvent(new MouseEvent('mousemove', {
                        bubbles: true,
                        cancelable: true,
                        clientX: bx,
                        clientY: currentY,
                        screenX: sp.screenX,
                        screenY: sp.screenY,
                        buttons: 1
                    }));
                    await new Promise(r => setTimeout(r, 55 + Math.floor(secureRandom() * 70)));
                    const bNowRaw = thumbElement.getAttribute('aria-valuenow');
                    const bNow = bNowRaw ? parseInt(bNowRaw, 10) : null;
                    if (bNow === targetAria) { lastAria = bNow; break; }
                    if (bNow !== null) lastAria = bNow;
                }
                break;
            }
        }
        
        // Редкие паузы
        if (secureRandom() < 0.035 && i - lastPause > 50) {
            await humanDelay('micro');
            lastPause = i;
        }
    }
    
    if (lastAria === null) {
        const fallbackDistance = (targetAria / (ariaMax - ariaMin || 100)) * maxDistance;
        const targetX = startX + Math.max(0, Math.min(maxDistance, fallbackDistance));
        updateMouseHistory(targetX, startY);
        const screenProps = getScreenProps();
        thumbElement.dispatchEvent(new MouseEvent('mousemove', {
            bubbles: true,
            cancelable: true,
            clientX: targetX,
            clientY: startY,
            screenX: screenProps.screenX,
            screenY: screenProps.screenY,
            buttons: 1
        }));
        await new Promise(r => setTimeout(r, 35));
    }
    
    await humanDelay('micro');
    
    const pointerUp = new PointerEvent('pointerup', {
        bubbles: true,
        cancelable: true,
        pointerId: getNextPointerId(),
        pointerType: 'mouse',
        clientX: mouseHistory.x,
        clientY: mouseHistory.y,
        pressure: 0
    });
    thumbElement.dispatchEvent(pointerUp);
    
    const mouseup = new MouseEvent('mouseup', {
        bubbles: true,
        cancelable: true,
        clientX: mouseHistory.x,
        clientY: mouseHistory.y,
        button: 0
    });
    thumbElement.dispatchEvent(mouseup);
    
    await new Promise(r => setTimeout(r, randomGaussian(600, 150)));
}

function simulateEnterPress(str) {
    const inputField = document.querySelector(str);
    if (inputField) {
        const event = new KeyboardEvent('keydown', {
            key: 'Enter',
            code: 'Enter',
            keyCode: 13,
            which: 13,
            bubbles: true,
            cancelable: true
        });
        inputField.dispatchEvent(event);
    } else {
        log("Can't click ENTER");
    }
}

function isVisible(el) {
    if (!el) return false;
    const style = window.getComputedStyle(el);
    return !!(el.offsetWidth || el.offsetHeight || el.getClientRects().length) 
        && style.visibility !== 'hidden' 
        && style.display !== 'none';
}

function log(...args) {
    console.log('[YandexCaptcha]', ...args);
}

function hasSrt(str, substr) {
    return str.indexOf(substr) !== -1;
}