let capUse = false;
let onoffk1 = false, onoffk2 = false;
let capUseCount = 0;
let lastProcessedButton = null;
let lastImageHash = null;
let mainTimerId = null; 


if (typeof isVisible !== 'function') {
    window.isVisible = function(elem) {
        if (!elem || !(elem instanceof Element)) {
            console.log('[GEETEST] isVisible: elem is not an element.');
            return false;
        }
        
        try {
            const style = window.getComputedStyle(elem);
            if (style.display === 'none') return false;
            if (style.visibility !== 'visible') return false;
            if (parseFloat(style.opacity) < 0.1) return false;
            
            const rect = elem.getBoundingClientRect();
            
            if (!rect || 
                !isFinite(rect.width) || !isFinite(rect.height) ||
                !isFinite(rect.left) || !isFinite(rect.top) ||
                !isFinite(rect.right) || !isFinite(rect.bottom)) {
                console.log('[GEETEST] isVisible: Invalid rect values');
                return false;
            }
            
            if (elem.offsetWidth + elem.offsetHeight + rect.height + rect.width === 0) {
                return false;
            }
            
            const elemCenter = {
                x: rect.left + (rect.width / 2),
                y: rect.top + (rect.height / 2)
            };
            
            if (!isFinite(elemCenter.x) || !isFinite(elemCenter.y)) {
                console.log('[GEETEST] isVisible: Invalid center coordinates');
                return false;
            }
            
            if (elemCenter.x < 0) return false;
            if (elemCenter.x > (document.documentElement.clientWidth || window.innerWidth)) return false;
            if (elemCenter.y < 0) return false;
            if (elemCenter.y > (document.documentElement.clientHeight || window.innerHeight)) return false;
            
            try {
                let pointContainer = document.elementFromPoint(elemCenter.x, elemCenter.y);
                if (!pointContainer) return false;
                
                do {
                    if (pointContainer === elem) return true;
                } while (pointContainer = pointContainer.parentNode);
                
                return false;
            } catch(e) {
                console.log('[GEETEST] isVisible: elementFromPoint error:', e);
                return (
                    rect.width > 0 &&
                    rect.height > 0 &&
                    rect.top < window.innerHeight &&
                    rect.bottom > 0 &&
                    rect.left < window.innerWidth &&
                    rect.right > 0
                );
            }
        } catch(error) {
            console.error('[GEETEST] isVisible error:', error);
            return false;
        }
    };
}

if (typeof delay !== 'function') {
    window.delay = function(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    };
}



if (typeof getBase64FromImageUrlYa !== 'function') {
    window.getBase64FromImageUrlYa = async function(url) {
        try {
            const res = await sendChrome.X('fetch', {"url": url, "op": {}});
            if (res && res.received && res.received !== 'WAIT' && !res.received.toString().includes('ERROR')) {
                return res.received;
            }
        } catch(e) {}
        return "";
    }
}


async function checkPluginEnabled() {
    try {
        if (typeof sendChrome === 'undefined' || !sendChrome.X) {
            return false;
        }
        
        const res = await sendChrome.X('get', {});
        
        if (!res || !res.received) {
            return false;
        }
        
        const settings = res.received;
        
        
        if (settings.onoff !== '1') {
            return false;
        }
        
        
        if (settings.onoff2 === '0') {
            return false;
        }
        
        
        if (settings.geetest !== '1') {
            return false;
        }
        
        return true;
        
    } catch (error) {
        console.error('[GEETEST] Error checking plugin status:', error);
        return false;
    }
}


function stopPlugin() {
    
    if (mainTimerId) {
        clearInterval(mainTimerId);
        mainTimerId = null;
    }
    
    
    capUse = false;
    lastProcessedButton = null;
    lastImageHash = null;
    onoffk1 = false;
    onoffk2 = false;
}


function startPlugin() {
    if (mainTimerId) {
        return;
    }
    
    initGeetestDetection();
}


if (typeof onoff !== 'function') {
    window.onoff = async function(asl) {
        const isEnabled = await checkPluginEnabled();
        
        
        return !isEnabled;
    };
}


if (typeof elClick !== 'function') {
    window.elClick = async function(element) {
        console.log("[GEETEST] Using fallback elClick");
        element.click();
    };
}


if (typeof elMove !== 'function') {
    window.elMove = async function(elementOrSelector, distX, distY = 0) {
        console.log("[GEETEST] Using improved elMove implementation");
        let element;
        
        if (typeof elementOrSelector === 'string') {
            element = document.querySelector(elementOrSelector);
        } else {
            element = elementOrSelector;
        }
        
        if (!element) {
            console.error("[GEETEST] Element not found for elMove");
            return false;
        }
        
        return await simulateDragImproved(element, distX, distY);
    };
}


if (typeof sendChrome === 'undefined') {
    window.sendChrome = {
        X: async function(type, data) {
            console.log("[GEETEST] Warning: sendChrome not defined, mock response");
            return { received: 'MOCK_RESPONSE' };
        }
    };
}


async function getBlobAsBase64(blobUrl) {
    try {
        const response = await fetch(blobUrl);
        if (!response.ok) {
            throw new Error(`Failed to fetch blob: ${response.status}`);
        }
        const blob = await response.blob();
        return new Promise((resolve, reject) => {
            const reader = new FileReader();
            reader.onloadend = () => {
                const base64 = reader.result.replace(/^data:image\/(png|jpeg|jpg|webp);base64,/, "");
                resolve(base64);
            };
            reader.onerror = () => reject(new Error("Failed to read blob"));
            reader.readAsDataURL(blob);
        });
    } catch (error) {
        console.error("Error converting blob to base64:", error);
        return null;
    }
}


async function getImageAsBase64(url) {
    if (!url) return null;
    
    try {
        if (url.startsWith("blob:")) {
            return await getBlobAsBase64(url);
        } else if (url.startsWith("data:")) {
            return url.replace(/^data:image\/(png|jpeg|jpg|webp);base64,/, "");
        } else {
            
            if (typeof sendChrome !== 'undefined' && sendChrome.X) {
                let fetchResponse = await sendChrome.X('fetch', { 'url': url });
                if (fetchResponse && fetchResponse.received) {
                    return fetchResponse.received;
                }
            }
            return null;
        }
    } catch (error) {
        console.error("Error getting image as base64:", error);
        return null;
    }
}


function createImageHash(base64Data) {
    if (!base64Data || base64Data.length < 100) return null;
    return base64Data.substring(0, 100) + base64Data.length;
}


function initGeetestDetection() {
    let maxuse = 3;
    let searchCount = 0;
    let processAttempts = 0;
    
    
    (async () => {
        const isEnabled = await checkPluginEnabled();
        if (!isEnabled) {
            return;
        }
        
        
        let initialWindow = await detectGeetestWindow();
        if(initialWindow) {
            console.log("[GEETEST] !!! Found already open captcha window on init!");
            capUse = true;
            
            let innerButton = await findButtonInWindow(initialWindow);
            if(innerButton) {
                console.log("[GEETEST] Found button inside initial window, clicking...");
                await clickGeetestButtonImproved(innerButton);
                await delay(1000);
            }
            
            setTimeout(async () => {
                let success = await eStart();
                if(!success) {
                    capUse = false;
                    console.log("[GEETEST] Failed to process initial captcha");
                }
            }, 1000);
        }
    })();

    
    mainTimerId = setInterval(async () => {
        searchCount++;
        
        
        const isPluginEnabled = await checkPluginEnabled();
        
        
        if (!isPluginEnabled) {
            
            if (capUse) {
                capUse = false;
                lastProcessedButton = null;
                lastImageHash = null;
            }
            
            
            return;
        }
        
        
        console.log(`[GEETEST] 🔄 Search iteration #${searchCount} (Plugin ENABLED)`);
        console.log(`[GEETEST] Current state: capUse=${capUse}, processAttempts=${processAttempts}`);
        
        
        let btnClick = document.querySelector('.geetest_btn_click');
        if(btnClick && isVisible(btnClick) && !capUse) {
            console.log("[GEETEST] Found .geetest_btn_click button!");
            capUse = true;
            await clickGeetestButtonImproved(btnClick);
            
            setTimeout(async () => {
                let success = await eStart();
                if(!success) {
                    capUse = false;
                    console.log("[GEETEST] Failed to process after btn_click");
                }
            }, 2000);
            
            return;
        }
        
        if(capUse) {
            console.log("[GEETEST] Captcha is being processed, skipping button search");
            
            let currentWindow = await detectGeetestWindow();
            if(currentWindow && processAttempts < 3) {
                console.log("[GEETEST] Active captcha window found");
                if(processAttempts > 0) {
                    processAttempts++;
                    let success = await eStart();
                    if(!success) {
                        capUse = false;
                        processAttempts = 0;
                        lastImageHash = null;
                        console.log("[GEETEST] Failed to process captcha, resetting");
                    }
                }
            } else if(!currentWindow) {
                capUse = false;
                processAttempts = 0;
                lastImageHash = null;
                console.log("[GEETEST] No captcha window found, resetting capUse");
            }
            
            return;
        }

        console.log("[GEETEST] Looking for Geetest buttons...");
        
        let geetestButtons = findGeetestButtonsImproved();
        
        if(geetestButtons.length > 0) {
            console.log(`[GEETEST] Found ${geetestButtons.length} potential Geetest buttons`);
            
            for(let i = 0; i < geetestButtons.length; i++) {
                let buttonInfo = geetestButtons[i];
                console.log(`[GEETEST] Button ${i + 1}/${geetestButtons.length} details:`, {
                    type: buttonInfo.type,
                    priority: buttonInfo.priority,
                    className: buttonInfo.button.className,
                    visible: isVisible(buttonInfo.button)
                });
                
                if(lastProcessedButton === buttonInfo.button) {
                    console.log("[GEETEST] Already processed this button, skipping");
                    continue;
                }
                
                if(isVisible(buttonInfo.button)) {
                    let className = buttonInfo.button.className || '';
                    let ariaLabel = buttonInfo.button.getAttribute('aria-label') || '';
                    
                    if(className.includes('close') || className.includes('logo') || 
                       ariaLabel.toLowerCase().includes('close') || ariaLabel.toLowerCase().includes('logo')) {
                        console.log("[GEETEST] Skipping close/logo button");
                        continue;
                    }
                    
                    console.log(`[GEETEST] >>> VISIBLE BUTTON FOUND! Attempting click...`);
                    
                    if(maxuse <= 0) { 
                        console.log("[GEETEST] Max attempts reached"); 
                        stopPlugin(); 
                        return;
                    }
                    maxuse--;
                    
                    lastProcessedButton = buttonInfo.button;
                    capUse = true;
                    
                    await clickGeetestButtonImproved(buttonInfo.button);
                    
                    setTimeout(async () => {
                        let success = await eStart();
                        if(!success) {
                            capUse = false;
                            console.log("[GEETEST] Failed to process captcha, resetting capUse");
                        }
                    }, 2000);
                    
                    break;
                }
            }
        }
        
        let currentWindow = await detectGeetestWindow();
        if(currentWindow && !capUse) {
            console.log("[GEETEST] Found active captcha window without button click!");
            capUse = true;
            processAttempts++;
            
            let innerButton = await findButtonInWindow(currentWindow);
            if(innerButton) {
                console.log("[GEETEST] Found button inside window, clicking...");
                await clickGeetestButtonImproved(innerButton);
                await delay(1000);
            }
            
            let success = await eStart();
            if(!success) {
                capUse = false;
                console.log("[GEETEST] Failed to process active captcha");
            }
        }
        
    }, 2000); 
}


function findGeetestButtonsImproved() {
    let buttons = [];
    let processedElements = new Set();
    
    console.log("[GEETEST] Starting button search...");
    
    
    try {
        let selector1 = '[class*="geetest"][aria-label]';
        let elements1 = document.querySelectorAll(selector1);
        console.log(`[GEETEST] Method 1 (${selector1}): found ${elements1.length} elements`);
        
        for(let element of elements1) {
            if(!processedElements.has(element)) {
                processedElements.add(element);
                buttons.push({
                    button: element,
                    type: "Direct: class*=geetest + aria-label",
                    priority: 1
                });
            }
        }
    } catch(e) {
        console.error("[GEETEST] Error in Method 1:", e);
    }
    
    
    try {
        let elements2 = document.querySelectorAll('[aria-label]');
        console.log(`[GEETEST] Method 2 (all aria-label): found ${elements2.length} elements`);
        
        let filtered2 = 0;
        for(let element of elements2) {
            if(!processedElements.has(element)) {
                let className = (element.className || '').toString().toLowerCase();
                let ariaLabel = (element.getAttribute('aria-label') || '').toLowerCase();
                
                let classPatterns = ['geetest', 'gt_', 'gee_', 'gt-', 'gee-', 'captcha'];
                let labelPatterns = ['geetest', 'captcha', 'verify', 'slide', 'drag', 'click', 'tap'];
                
                let matchesClass = classPatterns.some(pattern => className.includes(pattern));
                let matchesLabel = labelPatterns.some(pattern => ariaLabel.includes(pattern));
                
                if(matchesClass || matchesLabel) {
                    processedElements.add(element);
                    buttons.push({
                        button: element,
                        type: `Filtered: class(${matchesClass}) label(${matchesLabel})`,
                        priority: 2
                    });
                    filtered2++;
                }
            }
        }
        console.log(`[GEETEST] Method 2: filtered ${filtered2} relevant elements`);
    } catch(e) {
        console.error("[GEETEST] Error in Method 2:", e);
    }
    
    
    try {
        let selectors3 = [
            'button[class*="geetest"]',
            'div[class*="geetest"][onclick]',
            'div[class*="geetest"][role="button"]',
            '.geetest_radar_tip',
            '.geetest_radar_btn',
            '.geetest_btn_click',
            '.geetest_wrap',
            '[class*="geetest_btn"]',
            '[class*="geetest_radar"]'
        ];
        
        for(let selector of selectors3) {
            try {
                let elements = document.querySelectorAll(selector);
                if(elements.length > 0) {
                    console.log(`[GEETEST] Method 3 (${selector}): found ${elements.length} elements`);
                    
                    for(let element of elements) {
                        if(!processedElements.has(element)) {
                            processedElements.add(element);
                            buttons.push({
                                button: element,
                                type: `Selector: ${selector}`,
                                priority: 3
                            });
                        }
                    }
                }
            } catch(e) {
                console.error(`[GEETEST] Error with selector ${selector}:`, e);
            }
        }
    } catch(e) {
        console.error("[GEETEST] Error in Method 3:", e);
    }
    
    buttons.sort((a, b) => a.priority - b.priority);
    console.log(`[GEETEST] Total unique buttons found: ${buttons.length}`);
    return buttons;
}


async function clickGeetestButtonImproved(button) {
    console.log("[GEETEST] Starting advanced click sequence...");
    
    try {
        let rect = button.getBoundingClientRect();
        let centerX = rect.left + rect.width / 2;
        let centerY = rect.top + rect.height / 2;
        
        console.log("[GEETEST] Button coordinates:", { 
            centerX, 
            centerY, 
            width: rect.width, 
            height: rect.height 
        });
        
        if(button.scrollIntoView) {
            button.scrollIntoView({ behavior: 'smooth', block: 'center' });
            await delay(300);
        }
        
        if(button.focus) {
            console.log("[GEETEST] Focusing element...");
            button.focus();
            await delay(100);
        }
        
        console.log("[GEETEST] Simulating hover...");
        let hoverEvent = new MouseEvent('mouseover', {
            view: window,
            bubbles: true,
            cancelable: true,
            clientX: centerX,
            clientY: centerY
        });
        button.dispatchEvent(hoverEvent);
        await delay(100);
        
        console.log("[GEETEST] Native click...");
        button.click();
        await delay(100);
        
        console.log("[GEETEST] Click sequence completed!");
        
    } catch(error) {
        console.error("[GEETEST] Error during click sequence:", error);
    }
}


async function findButtonInWindow(window) {
    console.log("[GEETEST] Searching for button inside window...");
    
    const buttonSelectors = [
        '.geetest_radar_btn',
        '.geetest_radar_tip',
        '.geetest_btn',
        '.geetest_button',
        '.geetest_btn_click',
        '[class*="radar"]',
        '[class*="btn"]',
        'button',
        '[role="button"]'
    ];
    
    for(let selector of buttonSelectors) {
        let elements = window.querySelectorAll(selector);
        for(let element of elements) {
            if(isVisible(element)) {
                console.log(`[GEETEST] Found potential button in window: ${selector}`);
                
                let className = element.className || '';
                if(className.includes('refresh') || className.includes('result')) {
                    console.log("[GEETEST] Skipping refresh/result button");
                    continue;
                }
                
                return element;
            }
        }
    }
    
    return null;
}


async function detectGeetestWindow() {
    console.log("[GEETEST] Detecting Geetest window...");
    
    const windowSelectors = [
        '.geetest_box',
        '.geetest_window',
        '.geetest_popup',
        '.geetest_panel',
        '.geetest_holder',
        '[class*="geetest_box"]',
        '[class*="geetest_window"]',
        '.geetest_wrap',
        '.geetest_widget'
    ];
    
    for(let selector of windowSelectors) {
        let elements = document.querySelectorAll(selector);
        for(let element of elements) {
            if(isVisible(element)) {
                console.log(`[GEETEST] Found visible window with selector: ${selector}`, {
                    className: element.className,
                    id: element.id,
                    rect: element.getBoundingClientRect()
                });
                return element;
            }
        }
    }
    
    return null;
}


async function waitForCaptchaReady(geetestWindow, maxWait = 10000) {
    console.log("[GEETEST] Waiting for captcha to be ready...");
    
    let startTime = Date.now();
    while (Date.now() - startTime < maxWait) {
        
        let canvases = geetestWindow.querySelectorAll('canvas');
        for (let canvas of canvases) {
            if (canvas.width > 0 && canvas.height > 0) {
                try {
                    let ctx = canvas.getContext('2d');
                    let imageData = ctx.getImageData(0, 0, Math.min(canvas.width, 100), Math.min(canvas.height, 100));
                    let hasContent = imageData.data.some(pixel => pixel !== 0);
                    
                    if (hasContent) {
                        console.log("[GEETEST] Canvas has content, captcha is ready");
                        return true;
                    }
                } catch (e) {
                    
                }
            }
        }
        
        
        let bgElements = geetestWindow.querySelectorAll('[class*="geetest_bg"], [class*="slice"]');
        for (let element of bgElements) {
            let style = window.getComputedStyle(element);
            if (style.backgroundImage && style.backgroundImage !== 'none') {
                console.log("[GEETEST] Background image found, captcha is ready");
                return true;
            }
        }
        
        await delay(200);
    }
    
    console.log("[GEETEST] Timeout waiting for captcha to be ready");
    return false;
}

async function detectGeetestCanvas() {
    console.log("[GEETEST] Detecting Geetest canvas...");
    
    const canvasSelectors = [
        '.geetest_canvas canvas',
        '.geetest_canvas_bg canvas',
        '.geetest_canvas_fullbg canvas',
        '.geetest_canvas_slice canvas',
        '[class*="geetest"] canvas',
        'canvas'
    ];
    
    let canvasElements = [];
    
    for(let selector of canvasSelectors) {
        let elements = document.querySelectorAll(selector);
        for(let canvas of elements) {
            if(isVisible(canvas) && canvas.width > 0 && canvas.height > 0) {
                console.log(`[GEETEST] Found canvas with selector: ${selector}`, {
                    width: canvas.width,
                    height: canvas.height,
                    parent: canvas.parentElement?.className
                });
                canvasElements.push(canvas);
            }
        }
    }
    
    console.log(`[GEETEST] Total canvas elements on page: ${canvasElements.length}`);
    
    
    const bgElements = document.querySelectorAll('.geetest_bg, .geetest_slice_bg, [class*="geetest_item"]');
    for(let element of bgElements) {
        let style = window.getComputedStyle(element);
        if(style.backgroundImage && style.backgroundImage !== 'none') {
            console.log("[GEETEST] Found element with background-image:", {
                className: element.className,
                backgroundImage: style.backgroundImage.substring(0, 100)
            });
        }
    }
    
    return canvasElements;
}





async function detectGeetestClick(geetestWindow) {
    try {
        if(!geetestWindow) return null;

        const tipSelectors = [
            '.geetest_ques_tips',
            '.geetest_tips',
            '.geetest_tips_text',
            '[class*="geetest_ques_tips"]'
        ];
        let tipEl = null;
        for (let s of tipSelectors) {
            const el = geetestWindow.querySelector(s);
            if (el && isVisible(el)) { tipEl = el; break; }
        }
        const tipText = tipEl ? (tipEl.innerText || tipEl.textContent || '').trim() : '';

        
        const imgSelectors = [
            '.geetest_item_img',
            '.geetest_table_box img',
            '.geetest_panel_box img',
            '.geetest_img',
            '[class*="geetest_item_img"]',
            'canvas'
        ];
        let imgEl = null;
        for (let s of imgSelectors) {
            const els = geetestWindow.querySelectorAll(s);
            for (let el of els) {
                if (el && isVisible(el)) {
                    
                    const r = el.getBoundingClientRect();
                    if (r.width >= 120 && r.height >= 120) { imgEl = el; break; }
                }
            }
            if (imgEl) break;
        }

        if (!imgEl) return null;

        
        const canvases = geetestWindow.querySelectorAll('canvas');
        if (canvases && canvases.length >= 2) return null;

        return { imgEl, tipText };
    } catch(e) {
        console.log('[GEETEST] detectGeetestClick error:', e);
        return null;
    }
}

async function eStart() {
    console.log("[GEETEST] eStart() called - starting captcha processing");
    
    
    const isEnabled = await checkPluginEnabled();
    if (!isEnabled) {
        console.log("[GEETEST] ❌ Plugin disabled during eStart, aborting");
        return false;
    }
    
    let attempts = 0;
    const maxAttempts = 10;
    
    while(attempts < maxAttempts) {
        attempts++;
        console.log(`[GEETEST] Waiting for captcha window... Attempt ${attempts}/${maxAttempts}`);
        
        
        const stillEnabled = await checkPluginEnabled();
        if (!stillEnabled) {
            return false;
        }
        
        let geetestWindow = await detectGeetestWindow();
        if(geetestWindow) {
            console.log("[GEETEST] Geetest window detected!");
            
            
            let isReady = await waitForCaptchaReady(geetestWindow);
            if (!isReady) {
                console.log("[GEETEST] Captcha not ready, trying next attempt...");
                continue;
            }
            
            await delay(1000); 
            
            
            
            
            let clickData = await detectGeetestClick(geetestWindow);
            if (clickData && clickData.imgEl) {
                console.log("[GEETEST] Detected click-based captcha, preparing image...");
                let b64 = "";
                try {
                    if (clickData.imgEl.tagName && clickData.imgEl.tagName.toLowerCase() === 'img') {
                        const src = clickData.imgEl.src || "";
                        if (src.startsWith("data:")) {
                            b64 = src.split(",")[1] || "";
                        } else if (src) {
                            
                            b64 = await getBase64FromImageUrlYa(src);
                        }
                    } else if (clickData.imgEl.toDataURL) {
                        b64 = clickData.imgEl.toDataURL().split(",")[1] || "";
                    }
                } catch(e) {
                    console.log("[GEETEST] Failed to extract click image:", e);
                }

                if (b64 && typeof sendChrome !== 'undefined' && sendChrome.X) {
                    const tip = clickData.tipText ? ("click " + clickData.tipText) : "click";
                    let response = await sendChrome.X('imgs', {
                        'click': "geetest",
                        'type': tip,
                        'url': b64
                    });

                    console.log("[GEETEST] API response (click):", response);

                    if(response && response.received && response.received !== false &&
                       response.received !== 'WAIT' && response.received !== 'ERROR' &&
                       !response.received.toString().includes('ERROR')) {

                        let ok = await processGeetestClickResponse(response.received, geetestWindow, clickData.imgEl);
                        if (ok) {
                            await delay(1200);
                            
                            try {
                                const okBtn = geetestWindow.querySelector('.geetest_commit, .geetest_submit, [class*="geetest_commit"]');
                                if (okBtn && isVisible(okBtn)) {
                                    await simulate(okBtn, "click");
                                }
                            } catch(e){}
                            return true;
                        }
                    }
                }
                
            }

let canvasElements = await detectGeetestCanvas();
            
            if(canvasElements.length > 0) {
                console.log(`[GEETEST] Found ${canvasElements.length} canvas elements`);
                
                
                let targetCanvas = canvasElements.reduce((largest, current) => {
                    return (current.width * current.height) > (largest.width * largest.height) ? current : largest;
                });
                
                try {
                    
                    let ctx = targetCanvas.getContext('2d');
                    let imageData = ctx.getImageData(0, 0, targetCanvas.width, targetCanvas.height);
                    let isEmpty = !imageData.data.some(pixel => pixel !== 0);
                    
                    if(isEmpty) {
                        console.log("[GEETEST] Canvas is empty, waiting for image to load...");
                        await delay(1000);
                        continue;
                    }
                    
                    let canvasData = targetCanvas.toDataURL("image/png");
                    if(canvasData && canvasData.length > 1000) {
                        let base64Data = canvasData.replace(/^data:image\/(png|jpeg|jpg|webp);base64,/, "");
                        
                        
                        let currentHash = createImageHash(base64Data);
                        if(currentHash === lastImageHash) {
                            console.log("[GEETEST] Same image as before, skipping...");
                            await delay(1000);
                            continue;
                        }
                        lastImageHash = currentHash;
                        
                        console.log("[GEETEST] Canvas captured, base64 length:", base64Data.length);
                        console.log("[GEETEST] Canvas dimensions:", targetCanvas.width, "x", targetCanvas.height);
                        
                        if(typeof sendChrome !== 'undefined' && sendChrome.X) {
                            let response = await sendChrome.X('imgs', {
                                'click': "geetest",
                                'type': "slider",
                                'url': base64Data
                            });
                            
                            console.log("[GEETEST] API response:", response);
                            
                            if(response && response.received && 
                               response.received !== false && 
                               response.received !== 'WAIT' && 
                               response.received !== 'ERROR' &&
                               !response.received.toString().includes('ERROR')) {
                                let success = await processGeetestResponse(response.received, geetestWindow);
                                
                                if(success) {
                                    console.log("[GEETEST] Successfully processed captcha!");
                                    return true;
                                } else {
                                    console.log("[GEETEST] Failed to process response, trying again...");
                                    lastImageHash = null; 
                                    await delay(2000);
                                    continue;
                                }
                            } else if(response && response.received === false) {
                                console.log("[GEETEST] API couldn't process image, trying again...");
                                lastImageHash = null; 
                                await delay(2000);
                                continue;
                            }
                        }
                    }
                } catch(error) {
                    console.error("[GEETEST] Error processing canvas:", error);
                }
            } else {
                console.log("[GEETEST] No canvas found, checking for background images (v4)...");
                
                
                const bgSelectors = [
                    '.geetest_bg',
                    '.geetest_slice_bg',
                    '[class*="geetest_bg"]',
                    '[class*="geetest_slice_bg"]'
                ];
                
                let bgElement = null;
                
                for(let selector of bgSelectors) {
                    let elements = document.querySelectorAll(selector);
                    for(let element of elements) {
                        let style = window.getComputedStyle(element);
                        if(style.backgroundImage && style.backgroundImage !== 'none') {
                            console.log(`[GEETEST] Found background image element: ${selector}`);
                            bgElement = element;
                            break;
                        }
                    }
                    if(bgElement) break;
                }
                
                if(bgElement) {
                    console.log("[GEETEST] Found background images!");
                    
                    try {
                        
                        let bgStyle = window.getComputedStyle(bgElement);
                        let bgUrl = bgStyle.backgroundImage.match(/url\(["']?([^"')]+)["']?\)/)?.[1];
                        
                        console.log("[GEETEST] Background URL:", bgUrl);
                        
                        if(bgUrl) {
                            console.log("[GEETEST] Fetching background image...");
                            
                            let base64Data = await getImageAsBase64(bgUrl);
                            
                            if(base64Data) {
                                
                                let currentHash = createImageHash(base64Data);
                                if(currentHash === lastImageHash) {
                                    console.log("[GEETEST] Same background image as before, skipping...");
                                    await delay(1000);
                                    continue;
                                }
                                lastImageHash = currentHash;
                                
                                console.log("[GEETEST] Background image fetched successfully, base64 length:", base64Data.length);
                                
                                let response = await sendChrome.X('imgs', {
                                    'click': "geetest",
                                    'type': "slider", 
                                    'url': base64Data
                                });
                                
                                console.log("[GEETEST] API response:", response);
                                
                                if(response && response.received && 
                                   response.received !== false && 
                                   response.received !== 'WAIT' && 
                                   !response.received.toString().includes('ERROR')) {
                                    let success = await processGeetestResponse(response.received, geetestWindow);
                                    
                                    if(success) {
                                        console.log("[GEETEST] Successfully processed captcha!");
                                        return true;
                                    } else {
                                        console.log("[GEETEST] Failed to process response, trying again...");
                                        lastImageHash = null; 
                                        await delay(2000);
                                        continue;
                                    }
                                } else {
                                    console.log("[GEETEST] Invalid API response, trying again...");
                                    lastImageHash = null; 
                                    await delay(2000);
                                    continue;
                                }
                            } else {
                                console.error("[GEETEST] Failed to fetch background image");
                            }
                        }
                    } catch(error) {
                        console.error("[GEETEST] Error processing background images:", error);
                    }
                }
                
                
                let clickItems = document.querySelectorAll('.geetest_item, [class*="geetest_ghost"]');
                if(clickItems.length > 0) {
                    console.log("[GEETEST] Found click-type captcha with", clickItems.length, "items");
                    
                }
            }
        }
        
        await delay(3000); 
    }
    
    console.log("[GEETEST] Failed to detect captcha window after all attempts");
    return false;
}


async function processGeetestResponse(response, geetestWindow) {
    console.log("[GEETEST] Processing response:", response);
    
    
    let res1 = response.replace(/[^0-9,]/g, "");
    res1 = res1.split(",")[0]; 
    res1 = parseInt(res1);
    
    console.log("[GEETEST] Extracted coordinates from response:", {
        originalResult: response,
        cleanedNumbers: response.replace(/[^0-9,]/g, ""),
        splitArray: response.replace(/[^0-9,]/g, "").split(","),
        selectedValue: res1,
        selectedIndex: 0
    });
    
    if(!isNaN(res1) && res1 > 0) {
        
        const sliderSelectors = [
            '.geetest_slider_button',
            '.geetest_btn',
            '[class*="geetest_btn"]',
            '.geetest_slider',
            '.geetest_track',
            '[class*="geetest_track"]',
            '.geetest_btn_slide',
            '.geetest_arrow',
            '[class*="geetest_arrow"]',
            
            '.geetest_btn_88d300b7', 
            '[class*="geetest_btn_"]'
        ];
        
        let sliderButton = null;
        
        
        if(geetestWindow) {
            for(let selector of sliderSelectors) {
                let elements = geetestWindow.querySelectorAll(selector);
                for(let element of elements) {
                    if(isVisible(element)) {
                        
                        let style = window.getComputedStyle(element);
                        let className = element.className || '';
                        
                        
                        if(style.cursor === 'pointer' || style.cursor === 'move' || 
                           element.getAttribute('role') === 'slider' ||
                           className.includes('btn') || 
                           className.includes('slider') ||
                           className.includes('arrow')) {
                            console.log(`[GEETEST] Found slider in window with selector: ${selector}`, {
                                className: className,
                                cursor: style.cursor
                            });
                            sliderButton = element;
                            break;
                        }
                    }
                }
                if(sliderButton) break;
            }
        }
        
        if(sliderButton) {
            console.log("[GEETEST] Moving slider...");
            
            
            let finalDistance = res1;
            
            
            const imgElement = geetestWindow.querySelector("img") || 
                              geetestWindow.querySelector("canvas");
            
            if (imgElement) {
                let imageNaturalWidth = imgElement.naturalWidth || imgElement.width;
                let imageDisplayWidth = imgElement.offsetWidth || imgElement.clientWidth || imgElement.width;
                
                
                const sliderTrack = sliderButton.parentElement || 
                                   geetestWindow.querySelector('.geetest_slider_track') ||
                                   geetestWindow.querySelector('[class*="geetest_track"]');
                
                if(sliderTrack && imageNaturalWidth > 0 && imageDisplayWidth > 0) {
                    const trackWidth = sliderTrack.offsetWidth || sliderTrack.clientWidth;
                    const buttonWidth = sliderButton.offsetWidth || sliderButton.clientWidth;
                    const maxSliderDistance = trackWidth - buttonWidth;
                    
                    
                    const scaleRatio = imageDisplayWidth / imageNaturalWidth;
                    let scaledDistance = res1 * scaleRatio;
                    
                    
                    if(scaledDistance > maxSliderDistance) {
                        scaledDistance = maxSliderDistance * 0.95; 
                    }
                    
                    finalDistance = Math.round(scaledDistance);
                    
                    console.log("[GEETEST] IMPROVED scaling calculation:", {
                        originalDistance: res1,
                        imageNaturalWidth: imageNaturalWidth,
                        imageDisplayWidth: imageDisplayWidth,
                        scaleRatio: scaleRatio,
                        trackWidth: trackWidth,
                        buttonWidth: buttonWidth,
                        maxSliderDistance: maxSliderDistance,
                        scaledDistance: scaledDistance,
                        finalDistance: finalDistance
                    });
                } else {
                    
                    const naturalHeight = imgElement.naturalHeight || imgElement.height;
                    const offsetHeight = imgElement.offsetHeight || imgElement.height;
                    
                    if(naturalHeight > 0 && offsetHeight > 0) {
                        finalDistance = Math.round(res1 / naturalHeight * offsetHeight);
                        console.log("[GEETEST] Fallback height scaling:", {
                            originalDistance: res1,
                            naturalHeight: naturalHeight,
                            offsetHeight: offsetHeight,
                            finalDistance: finalDistance
                        });
                    }
                }
            }
            
            
            finalDistance = finalDistance - 8; 
            
            
            if(finalDistance < 0) {
                finalDistance = Math.max(res1 * 0.8, 10); 
            }
            
            console.log("[GEETEST] Final distance for slider:", finalDistance);
            
            
            try {
                await elMove(sliderButton, finalDistance, 0);
                console.log("[GEETEST] Slider moved successfully with elMove");
            } catch(error) {
                console.log("[GEETEST] elMove failed, using fallback simulateDragImproved:", error);
                await simulateDragImproved(sliderButton, finalDistance);
            }
            
            await delay(2000);
            let result = await checkCaptchaResult();
            return result;
        } else {
            console.log("[GEETEST] Slider button not found");
            
            
            console.log("[GEETEST] Debugging - Looking for slider elements:");
            let allPotentialSliders = document.querySelectorAll('[class*="geetest"]');
            allPotentialSliders.forEach(el => {
                let style = window.getComputedStyle(el);
                if(style.cursor === 'pointer' || style.cursor === 'move' || el.className.includes('btn')) {
                    console.log("Potential slider:", {
                        className: el.className,
                        tagName: el.tagName,
                        cursor: style.cursor,
                        visible: isVisible(el),
                        rect: el.getBoundingClientRect()
                    });
                }
            });
            return false;
        }
    } else {
        console.log("[GEETEST] Invalid distance value:", res1);
        return false;
    }
}

async function processGeetestClickResponse(response, geetestWindow, imgEl) {
    console.log("[GEETEST] Processing click-response:", response);

    
    let nums = (response || "").toString().replace(/[^0-9,]/g, "").split(",").filter(v => v !== "");
    nums = nums.map(v => parseInt(v)).filter(v => !isNaN(v));

    if (!imgEl || nums.length === 0) {
        console.log("[GEETEST] Click-response invalid or missing image element");
        return false;
    }

    const rect = imgEl.getBoundingClientRect();
    const clicks = [];

    if (nums.length >= 2 && nums.length % 2 === 0) {
        
        for (let i = 0; i < nums.length; i += 2) {
            clicks.push({ x: nums[i], y: nums[i+1] });
        }
    } else {
        
        const grid = 3;
        const cellW = rect.width / grid;
        const cellH = rect.height / grid;
        for (let n of nums) {
            const idx = Math.max(1, n) - 1;
            const cx = (idx % grid) * cellW + cellW/2;
            const cy = Math.floor(idx / grid) * cellH + cellH/2;
            clicks.push({ x: Math.round(cx), y: Math.round(cy) });
        }
    }

    console.log("[GEETEST] Click points:", clicks);

    for (let p of clicks) {
        const clientX = rect.left + p.x;
        const clientY = rect.top + p.y;
        try {
            await simulateMouseEvent(imgEl, "mousemove", clientX, clientY);
            await delay(80);
            await simulateMouseEvent(imgEl, "mousedown", clientX, clientY);
            await delay(40);
            await simulateMouseEvent(imgEl, "mouseup", clientX, clientY);
            await delay(150);
            await simulateMouseEvent(imgEl, "click", clientX, clientY);
            await delay(400);
        } catch(e) {
            console.log("[GEETEST] click simulate error:", e);
        }
    }

    return true;
}




async function simulateDragImproved(element, distance, dy = 0) {
    console.log("[GEETEST] Simulating improved drag...");
    
    try {
        let rect = element.getBoundingClientRect();
        let startX = rect.left + rect.width / 2;
        let startY = rect.top + rect.height / 2;
        let endX = startX + distance;
        let endY = startY + dy;
        
        console.log("[GEETEST] Drag coordinates:", {
            startX, startY, endX, endY, distance
        });
        
        
        if (element.focus) {
            element.focus();
            await delay(50);
        }
        
        
        let mouseDownEvent = new MouseEvent('mousedown', {
            view: window,
            bubbles: true,
            cancelable: true,
            clientX: startX,
            clientY: startY,
            buttons: 1,
            button: 0
        });
        element.dispatchEvent(mouseDownEvent);
        await delay(100);
        
        
        let steps = Math.max(20, Math.abs(distance) / 5);
        for(let i = 1; i <= steps; i++) {
            let progress = i / steps;
            
            
            let variance = (Math.random() - 0.5) * 2;
            let currentX = startX + (distance * progress) + variance;
            let currentY = startY + (dy * progress) + variance;
            
            let mouseMoveEvent = new MouseEvent('mousemove', {
                view: window,
                bubbles: true,
                cancelable: true,
                clientX: currentX,
                clientY: currentY,
                buttons: 1,
                button: 0
            });
            
            element.dispatchEvent(mouseMoveEvent);
            document.dispatchEvent(mouseMoveEvent);
            
            await delay(10 + Math.random() * 20);
        }
        
        
        let finalMoveEvent = new MouseEvent('mousemove', {
            view: window,
            bubbles: true,
            cancelable: true,
            clientX: endX,
            clientY: endY,
            buttons: 1,
            button: 0
        });
        element.dispatchEvent(finalMoveEvent);
        document.dispatchEvent(finalMoveEvent);
        await delay(100);
        
        
        let mouseUpEvent = new MouseEvent('mouseup', {
            view: window,
            bubbles: true,
            cancelable: true,
            clientX: endX,
            clientY: endY,
            buttons: 0,
            button: 0
        });
        element.dispatchEvent(mouseUpEvent);
        document.dispatchEvent(mouseUpEvent);
        
        console.log(`[GEETEST] Improved drag completed: ${distance}px`);
        return true;
        
    } catch(error) {
        console.error("[GEETEST] Error in simulateDragImproved:", error);
        return false;
    }
}

async function checkCaptchaResult() {
    console.log("[GEETEST] Checking captcha result...");
    
    await delay(2000); 
    
    const successSelectors = [
        '.geetest_success',
        '[class*="success"]',
        '.geetest_result.geetest_success',
        '[class*="geetest"][class*="success"]',
        '.geetest_success_tips',
        '.geetest_success_radar'
    ];
    
    for(let selector of successSelectors) {
        let successElement = document.querySelector(selector);
        if(successElement && isVisible(successElement)) {
            console.log("[GEETEST] SUCCESS! Captcha solved successfully with selector:", selector);
            capUse = false;
            lastProcessedButton = null;
            lastImageHash = null; 
            return true;
        }
    }
    
    
    let geetestWindow = await detectGeetestWindow();
    if(!geetestWindow) {
        console.log("[GEETEST] Captcha window disappeared - likely success");
        capUse = false;
        lastProcessedButton = null;
        lastImageHash = null; 
        return true;
    }
    
    
    let sliderSelectors = [
        '.geetest_slider_button',
        '.geetest_btn',
        '[class*="geetest_btn"]'
    ];
    
    for(let selector of sliderSelectors) {
        let slider = document.querySelector(selector);
        if(slider) {
            let rect = slider.getBoundingClientRect();
            
            if(rect.left < 50) {
                console.log("[GEETEST] Slider moved back, captcha likely failed");
                lastImageHash = null; 
                return false;
            }
        }
    }
    
    console.log("[GEETEST] No clear success or failure indicators found");
    return false;
}


function detectGeetestPresence() {
    
    if(window.initGeetest || window.initGeetest4 || window.Geetest) {
        return true;
    }
    
    
    let scripts = document.querySelectorAll('script[src*="geetest"], script[src*="gt.js"]');
    if(scripts.length > 0) {
        return true;
    }
    
    
    let elements = document.querySelectorAll('[class*="geetest"], [id*="geetest"], [data-geetest]');
    if(elements.length > 0) {
        return true;
    }
    
    return false;
}


function startGlobalMonitoring() {
    
    setInterval(async () => {
        const isEnabled = await checkPluginEnabled();
        
        if (!isEnabled && mainTimerId) {
            
            stopPlugin();
        } else if (isEnabled && !mainTimerId) {
            
            startPlugin();
        }
    }, 3000);
}




if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', async () => {
        if(detectGeetestPresence()) {
            
            startGlobalMonitoring();
            
            
            const isEnabled = await checkPluginEnabled();
            if (isEnabled) {
                startPlugin();
            }
        } else {
            
            let checkInterval = setInterval(async () => {
                if(detectGeetestPresence()) {
                    clearInterval(checkInterval);
                    
                    startGlobalMonitoring();
                    
                    const isEnabled = await checkPluginEnabled();
                    if (isEnabled) {
                        startPlugin();
                    }
                }
            }, 2000);
        }
    });
} else {
    if(detectGeetestPresence()) {
        
        startGlobalMonitoring();
        
        
        (async () => {
            const isEnabled = await checkPluginEnabled();
            if (isEnabled) {
                startPlugin();
            }
        })();
    } else {
        
        let checkInterval = setInterval(async () => {
            if(detectGeetestPresence()) {
                clearInterval(checkInterval);
                
                startGlobalMonitoring();
                
                const isEnabled = await checkPluginEnabled();
                if (isEnabled) {
                    startPlugin();
                }
            }
        }, 2000);
    }
}