/**/
// Многоязычные тексты
const translations = {
    en: {
        title: "Plugin Disabled",
        message: "Please enable the plugin to access captcha solving features.",
        button: "Enable Plugin",
        headerBalance: "Your balance:",
        topUp: "Top up",
        apiPlaceholder: "Enter your API KEY",
        reloadBtn: "🔄 Reload",
        statusBtn: "📊 Status",
        captchaTypes: "Type of captchas:",
        contact: "Have a question?",
        pluginOn: "Plugin is ON!",
        hcaptchaInDevelopment: "🚧 In Development\n\nhCaptcha support is currently being developed.\nThis feature will be available in a future update.",
        awsInDevelopment: "🚧 In Development\n\nAWS Captcha support is currently being developed.\nThis feature will be available in a future update."
    },
    ru: {
        title: "Плагин отключен",
        message: "Пожалуйста, включите плагин для доступа к функциям решения капчи.",
        button: "Включить плагин",
        headerBalance: "Ваш баланс:",
        topUp: "Пополнить",
        apiPlaceholder: "Введите ваш API ключ",
        reloadBtn: "🔄 Перезагрузить",
        statusBtn: "📊 Статус",
        captchaTypes: "Типы капч:",
        contact: "Есть вопрос?",
        pluginOn: "Плагин включен!",
        hcaptchaInDevelopment: "🚧 В разработке\n\nПоддержка hCaptcha сейчас разрабатывается.\nЭта функция будет доступна в будущих обновлениях.",
        awsInDevelopment: "🚧 В разработке\n\nПоддержка AWS Captcha сейчас разрабатывается.\nЭта функция будет доступна в будущих обновлениях."
    }
};

// Определение языка браузера
function getBrowserLanguage() {
    try {
        const lang = chrome.i18n.getUILanguage() || navigator.language || navigator.languages[0] || 'en';
        const langCode = lang.toLowerCase().split('-')[0];
        return langCode === 'ru' ? 'ru' : 'en';
    } catch (error) {
        const lang = navigator.language || navigator.languages[0] || 'en';
        const langCode = lang.toLowerCase().split('-')[0];
        return langCode === 'ru' ? 'ru' : 'en';
    }
}

// Функция для обработки клика по hCaptcha
function handleHCaptchaClick(event) {
    // Предотвращаем стандартное поведение
    event.preventDefault();
    event.stopPropagation();
    
    // Получаем текущий язык
    const lang = getBrowserLanguage();
    const t = translations[lang];
    
    // Показываем сообщение о разработке
    showNotification(t.hcaptchaInDevelopment, 'warning', 5000);
    
    // Сбрасываем состояние чекбокса, если он был включен
    const hcaptchaCheckbox = document.querySelector('.cl_hcap');
    if (hcaptchaCheckbox && hcaptchaCheckbox.checked) {
        hcaptchaCheckbox.checked = false;
        console.log('🚧 hCaptcha отключен - функция в разработке');
    }
}

// Функция для обработки клика по AWS Captcha
function handleAWSClick(event) {
    // Предотвращаем стандартное поведение
    event.preventDefault();
    event.stopPropagation();
    
    // Получаем текущий язык
    const lang = getBrowserLanguage();
    const t = translations[lang];
    
    // Показываем сообщение о разработке
    showNotification(t.awsInDevelopment, 'warning', 5000);
    
    // Сбрасываем состояние чекбокса, если он был включен
    const awsCheckbox = document.querySelector('.cl_aws');
    if (awsCheckbox && awsCheckbox.checked) {
        awsCheckbox.checked = false;
        console.log('🚧 AWS Captcha отключен - функция в разработке');
    }
}

// Применение переводов
function applyTranslations() {
    const lang = getBrowserLanguage();
    const t = translations[lang];
    
    // Проверяем существование элементов overlay
    const inactiveTitle = document.getElementById('inactiveTitle');
    const inactiveMessage = document.getElementById('inactiveMessage');
    const activateButton = document.getElementById('activateButton');
    
    if (inactiveTitle) inactiveTitle.textContent = t.title;
    if (inactiveMessage) inactiveMessage.textContent = t.message;
    if (activateButton) activateButton.textContent = t.button;
    
    // Обновляем другие тексты интерфейса
    const balanceEl = document.querySelector('.block_key_header_name');
    if (balanceEl) {
        const currentBalance = balanceEl.querySelector('.balans')?.textContent || '0';
        balanceEl.innerHTML = `${t.headerBalance} <span class="balans">${currentBalance}</span>`;
    }
    
    const topUpLink = document.querySelector('.block_key_header_link a');
    if (topUpLink) topUpLink.textContent = t.topUp;
    
    const apiInput = document.querySelector('.block_key_input');
    if (apiInput) apiInput.placeholder = t.apiPlaceholder;
    
    const reloadBtn = document.getElementById('reloadConfigBtn');
    if (reloadBtn) reloadBtn.title = t.reloadBtn.replace('🔄 ', '');
    
    const statusBtn = document.getElementById('configStatusBtn');
    if (statusBtn) statusBtn.title = t.statusBtn.replace('📊 ', '');
    
    const contactLink = document.querySelector('.repot_contact a');
    if (contactLink) contactLink.textContent = t.contact;
    
    const pluginOnMsg = document.querySelector('.not_show_block');
    if (pluginOnMsg && !pluginOnMsg.querySelector('img')) {
        pluginOnMsg.textContent = t.pluginOn;
    }
}

// Обновляем CSS для мультиязычности
function updateCSSForLanguage() {
    const lang = getBrowserLanguage();
    const t = translations[lang];
    
    // Удаляем предыдущий стиль если есть
    const existingStyle = document.getElementById('lang-style');
    if (existingStyle) {
        existingStyle.remove();
    }
    
    // Создаем и добавляем стиль для псевдоэлемента
    const style = document.createElement('style');
    style.id = 'lang-style';
    style.textContent = `
        .auto_ch::before {
            content: "${t.captchaTypes}";
            display: block;
            font-weight: 600;
            color: #374151;
            margin-bottom: 8px;
            font-size: 12px;
        }
    `;
    document.head.appendChild(style);
}

// Функция активации плагина
function activatePlugin() {
    const mainToggle = document.querySelector(".cl_onoff");
    const wrapper = document.getElementById('pluginWrapper') || document.querySelector('.wrapper');
    const overlay = document.getElementById('inactiveOverlay');
    
    if (mainToggle) {
        // Включаем главный переключатель
        mainToggle.checked = true;
        
        // Скрываем overlay и убираем класс неактивности
        if (overlay) overlay.classList.add('hidden');
        if (wrapper) wrapper.classList.remove('plugin-inactive');
        
        // Имитируем событие изменения для сохранения состояния
        mainToggle.dispatchEvent(new Event('change'));
        
        console.log('Plugin activated via overlay button');
    }
}

// Функция проверки состояния плагина
function checkPluginState() {
    const mainToggle = document.querySelector(".cl_onoff");
    const wrapper = document.getElementById('pluginWrapper') || document.querySelector('.wrapper');
    const overlay = document.getElementById('inactiveOverlay');
    
    if (!mainToggle || !overlay || !wrapper) {
        console.warn('Required elements for plugin state check not found');
        return;
    }
    
    if (!mainToggle.checked) {
        // Плагин неактивен - показываем overlay
        overlay.classList.remove('hidden');
        wrapper.classList.add('plugin-inactive');
    } else {
        // Плагин активен - скрываем overlay
        overlay.classList.add('hidden');
        wrapper.classList.remove('plugin-inactive');
    }
}

// Создание overlay элемента если его нет
function createInactiveOverlay() {
    // Проверяем, есть ли уже overlay
    if (document.getElementById('inactiveOverlay')) {
        return;
    }
    
    const wrapper = document.querySelector('.wrapper');
    if (!wrapper) {
        console.warn('Wrapper not found, cannot create inactive overlay');
        return;
    }
    
    // Добавляем ID к wrapper если его нет
    if (!wrapper.id) {
        wrapper.id = 'pluginWrapper';
    }
    
    // Создаем overlay
    const overlay = document.createElement('div');
    overlay.className = 'plugin-inactive-overlay hidden';
    overlay.id = 'inactiveOverlay';
    
    overlay.innerHTML = `
        <div class="inactive-icon"></div>
        <div class="inactive-title" id="inactiveTitle">Plugin Disabled</div>
        <div class="inactive-message" id="inactiveMessage">
            Please enable the plugin to access captcha solving features.
        </div>
        <button class="activate-button" id="activateButton" onclick="activatePlugin()">
            Enable Plugin
        </button>
    `;
    
    // Добавляем CSS стили для overlay
    const style = document.createElement('style');
    style.id = 'inactive-overlay-styles';
    style.textContent = `
        .plugin-inactive-overlay {
            position: absolute;
            top: 0;
            left: 0;
            width: 100%;
            height: 100%;
            background: linear-gradient(135deg, 
                rgba(255, 255, 255, 0.98) 0%, 
                rgba(248, 250, 252, 0.98) 100%);
            backdrop-filter: blur(15px);
            z-index: 1000;
            display: flex;
            flex-direction: column;
            align-items: center;
            justify-content: center;
            text-align: center;
            padding: 20px;
            border-radius: 8px;
            box-shadow: 0 20px 40px rgba(0, 0, 0, 0.1);
        }

        .plugin-inactive-overlay.hidden {
            display: none !important;
        }

        .inactive-icon {
            width: 60px;
            height: 60px;
            background: linear-gradient(135deg, #FF6B6B, #FF8E8E);
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            margin-bottom: 16px;
            box-shadow: 0 8px 24px rgba(255, 107, 107, 0.3);
            animation: pulse 2s infinite;
        }

        .inactive-icon::before {
            content: "!";
            font-size: 28px;
            font-weight: 700;
            color: white;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.2);
        }

        @keyframes pulse {
            0%, 100% {
                transform: scale(1);
                box-shadow: 0 8px 24px rgba(255, 107, 107, 0.3);
            }
            50% {
                transform: scale(1.05);
                box-shadow: 0 12px 32px rgba(255, 107, 107, 0.4);
            }
        }

        .inactive-title {
            font-size: 16px;
            font-weight: 700;
            color: #1F2937;
            margin-bottom: 8px;
            background: linear-gradient(135deg, #1F2937, #374151);
            -webkit-background-clip: text;
            -webkit-text-fill-color: transparent;
            background-clip: text;
        }

        .inactive-message {
            font-size: 12px;
            color: #6B7280;
            line-height: 1.4;
            margin-bottom: 16px;
            max-width: 200px;
        }

        .activate-button {
            background: linear-gradient(135deg, #3598dc, #43a4e8);
            color: white;
            border: none;
            padding: 8px 16px;
            border-radius: 20px;
            font-size: 11px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s cubic-bezier(0.4, 0, 0.2, 1);
            box-shadow: 0 4px 12px rgba(53, 152, 220, 0.3);
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .activate-button:hover {
            transform: translateY(-2px);
            box-shadow: 0 6px 20px rgba(53, 152, 220, 0.4);
            background: linear-gradient(135deg, #43a4e8, #3598dc);
        }

        .activate-button:active {
            transform: translateY(0px);
            box-shadow: 0 2px 8px rgba(53, 152, 220, 0.3);
        }

        .wrapper.plugin-inactive .block2:not(.header) {
            opacity: 0.3;
            pointer-events: none;
            filter: blur(1px);
            transition: all 0.3s ease;
        }

        .wrapper.plugin-inactive .repot_contact {
            opacity: 0.3;
            pointer-events: none;
            filter: blur(1px);
            transition: all 0.3s ease;
        }
    `;
    
    // Добавляем стили в head если их еще нет
    if (!document.getElementById('inactive-overlay-styles')) {
        document.head.appendChild(style);
    }
    
    // Вставляем overlay в wrapper
    wrapper.appendChild(overlay);
    
    console.log('Inactive overlay created');
}

// Глобальные переменные для управления данными
let cachedData = null;
let isDataLoading = false;
let preloadPromise = null;
let configManagerReady = false;

// Новые переменные для системы повторных попыток
let dataLoadAttempts = 0;
const MAX_LOAD_ATTEMPTS = 3;
const LOAD_RETRY_DELAY = 300;

// Основная функция предзагрузки данных
async function preloadData() {
	if (isDataLoading && preloadPromise) {
		return preloadPromise;
	}
	
	isDataLoading = true;
	
	preloadPromise = (async () => {
		try {
			console.log('Начинаем предзагрузку данных...');
			
			// Ждем готовности ConfigManager
			await waitForConfigManager();
			
			// Загружаем внешнюю конфигурацию
			await loadExternalConfig();
			
			// Пытаемся получить данные из background script несколько раз
			let res = null;
			for (let attempt = 0; attempt < MAX_LOAD_ATTEMPTS; attempt++) {
				res = await sendChrome.X('get', {});
				
				if (res && res.received) {
					break;
				}
				
				console.warn(`Попытка ${attempt + 1}/${MAX_LOAD_ATTEMPTS} получить данные неудачна, повторяем...`);
				if (attempt < MAX_LOAD_ATTEMPTS - 1) {
					await delay(LOAD_RETRY_DELAY * (attempt + 1));
				}
			}
			
			if (res && res.received) {
				cachedData = res;
				console.log('Данные предзагружены:', cachedData);
			} else {
				console.warn('Не удалось получить данные после всех попыток, используем данные по умолчанию');
				// Создаем минимальный набор данных по умолчанию
				cachedData = {
					received: {
						apikey: '',
						onoff: '0',
						onoff2: '0',
						balans: '0'
					}
				};
			}
			
			// Отправляем сигнал о готовности данных в background script
			try {
				chrome.runtime.sendMessage({
					action: 'dataPreloaded', 
					data: cachedData,
					timestamp: Date.now()
				});
			} catch (error) {
				console.warn('Не удалось отправить сигнал о предзагрузке:', error);
			}
			
			return cachedData;
			
		} catch (error) {
			console.error('Ошибка предзагрузки данных:', error);
			// Возвращаем базовые данные в случае ошибки
			return {
				received: {
					apikey: '',
					onoff: '0',
					onoff2: '0',
					balans: '0'
				}
			};
		} finally {
			isDataLoading = false;
		}
	})();
	
	return preloadPromise;
}

// Улучшенное ожидание загрузки ConfigManager
async function waitForConfigManager() {
	let attempts = 0;
	const maxAttempts = 50;
	
	while (!window.configManager && attempts < maxAttempts) {
		await delay(50);
		attempts++;
		
		if (attempts === 10 && !window.configManager) {
			try {
				await loadConfigManagerScript();
			} catch (error) {
				console.warn('Не удалось загрузить ConfigManager:', error);
			}
		}
	}
	
	configManagerReady = !!window.configManager;
	
	if (!configManagerReady) {
		console.error('ConfigManager не загружен после', maxAttempts, 'попыток');
	} else {
		console.log('ConfigManager готов');
	}
	
	return configManagerReady;
}

// Загрузка скрипта ConfigManager
async function loadConfigManagerScript() {
	return new Promise((resolve, reject) => {
		const script = document.createElement('script');
		script.src = chrome.runtime.getURL('config-manager.js');
		script.onload = resolve;
		script.onerror = reject;
		(document.head || document.documentElement).appendChild(script);
	});
}

// Новая функция для проверки валидности данных
function isDataValid(data) {
	if (!data || typeof data !== 'object') {
		return false;
	}
	
	// Проверяем наличие ключевых полей
	const requiredFields = ['onoff', 'onoff2'];
	for (const field of requiredFields) {
		if (!(field in data)) {
			return false;
		}
	}
	
	return true;
}

// Новая функция для повторной попытки загрузки данных
async function retryDataLoad() {
	dataLoadAttempts++;
	
	if (dataLoadAttempts > MAX_LOAD_ATTEMPTS) {
		console.error('Превышено максимальное количество попыток загрузки данных');
		// Используем минимальные данные по умолчанию
		const defaultData = {
			apikey: '',
			onoff: '0',
			onoff2: '0',
			balans: '0'
		};
		populateUI(defaultData);
		return false;
	}
	
	console.log(`Повторная попытка загрузки данных (${dataLoadAttempts}/${MAX_LOAD_ATTEMPTS})`);
	await delay(LOAD_RETRY_DELAY * dataLoadAttempts);
	
	try {
		const res = await sendChrome.X('get', {});
		if (res && res.received && isDataValid(res.received)) {
			cachedData = res;
			populateUI(res.received);
			console.log('Данные успешно загружены при повторной попытке');
			dataLoadAttempts = 0; // Сбрасываем счетчик при успехе
			return true;
		} else {
			return await retryDataLoad(); // Рекурсивный вызов
		}
	} catch (error) {
		console.error('Ошибка при повторной загрузке:', error);
		return await retryDataLoad(); // Рекурсивный вызов
	}
}

// Основная инициализация при загрузке DOM
document.addEventListener('DOMContentLoaded', async function() {
	console.log('DOMContentLoaded - начинаем инициализацию');
	
	// Создаем overlay для неактивного плагина
	createInactiveOverlay();
	
	// Применяем переводы
	applyTranslations();
	updateCSSForLanguage();
	
	// Показываем индикатор загрузки
	showLoadingIndicator(true);
	
	try {
		// Сброс счетчика попыток при каждом открытии popup
		dataLoadAttempts = 0;
		
		// Ждем завершения предзагрузки или запускаем её
		if (!cachedData && !isDataLoading) {
			await preloadData();
		} else if (isDataLoading && preloadPromise) {
			await preloadPromise;
		}
		
		// Быстрая загрузка из кэша с проверкой валидности
		let loadSuccess = false;
		if (cachedData) {
			console.log('Загружаем из кэша');
			loadSuccess = await loadFromCache();
		}
		
		// Если загрузка из кэша не удалась, пытаемся загрузить напрямую
		if (!loadSuccess) {
			console.log('Кэш недоступен или некорректен, загружаем данные напрямую');
			await retryDataLoad();
		}
		
		// Настраиваем обработчики событий
		setupEventHandlers();
		
		// Обновляем данные в фоне (без блокировки UI) с задержкой
		setTimeout(() => {
			refreshDataInBackground();
		}, 500);
		
		// Обновляем статус конфигурации
		await updateConfigStatus();
		
		// Проверяем состояние плагина после загрузки данных с небольшой задержкой
		setTimeout(() => {
			checkPluginState();
		}, 100);
		
	} catch (error) {
		console.error('Ошибка инициализации popup:', error);
		showNotification('Ошибка загрузки данных', 'error');
		
		// В случае критической ошибки используем данные по умолчанию
		const defaultData = {
			apikey: '',
			onoff: '0',
			onoff2: '0',
			balans: '0'
		};
		populateUI(defaultData);
	} finally {
		// Скрываем индикатор загрузки
		showLoadingIndicator(false);
	}
});

// Настройка всех обработчиков событий
function setupEventHandlers() {
	console.log('🔧 Настраиваем обработчики событий...');
	
	// Обработчик для поля API ключа
	const sourceEl = document.querySelector(".block_key_input");
	
	const inputHandler = async function(e) {
		const apiKey = e.target.value;
		try {
			const res = await sendChrome.X('set', {"n":"apikey", 'v': apiKey});
			await delay(200);
			console.log('API key updated:', res);
			
			await load();
			await updateConfigStatus();
			
			// Обновляем кэш
			cachedData = await sendChrome.X('get', {});
		} catch (error) {
			console.error('Ошибка обновления API ключа:', error);
			showNotification('Ошибка обновления API ключа', 'error');
		}
	};
	
	if (sourceEl) {
		sourceEl.addEventListener('input', debounce(inputHandler, 500));
		sourceEl.addEventListener('propertychange', debounce(inputHandler, 500));
		console.log('✅ Обработчики API ключа добавлены');
	}
	
	// Обработчики для чекбоксов (добавлен .cl_geetest)
	const checkboxes = document.querySelectorAll('.cl_onoff,.cl_tiktok,.cl_oth,.cl_hcap,.cl_recap,.cl_geetest,.cl_funcap,.cl_binance,.cl_aws');
	console.log('🔍 Найдено чекбоксов:', checkboxes.length);
	
	for (let i = 0; i < checkboxes.length; i++) {
		const checkbox = checkboxes[i];
		console.log(`📋 Добавляем обработчик для: ${checkbox.name || checkbox.className}`);
		
		// Специальная обработка для hCaptcha
		if (checkbox.classList.contains('cl_hcap')) {
			checkbox.addEventListener("change", function(e) {
				// Если пытаются включить hCaptcha - показываем сообщение и отключаем
				if (this.checked) {
					handleHCaptchaClick(e);
				}
			});
			console.log('🚧 Специальный обработчик для hCaptcha добавлен');
		}
		// Специальная обработка для AWS Captcha
		else if (checkbox.classList.contains('cl_aws')) {
			checkbox.addEventListener("change", function(e) {
				// Если пытаются включить AWS - показываем сообщение и отключаем
				if (this.checked) {
					handleAWSClick(e);
				}
			});
			console.log('🚧 Специальный обработчик для AWS Captcha добавлен');
		} else {
			// Стандартная обработка для остальных чекбоксов
			checkbox.addEventListener("change", async function(e) {
				const name = this.name;
				const value = this.checked ? '1' : '';
				
				console.log(`🔄 Изменение настройки: ${name} = ${value}`);
				
				try {
					await sendChrome.X('set', {"n": name, 'v': value});
					await delay(200);
					await load();
					await updateConfigStatus();
					
					// Проверяем состояние плагина после изменения главного переключателя
					if (name === 'onoff') {
						console.log('🔄 Главный переключатель изменен, проверяем состояние...');
						checkPluginState();
					}
					
					// Обновляем кэш
					cachedData = await sendChrome.X('get', {});
				} catch (error) {
					console.error('Ошибка обновления настройки:', error);
					showNotification('Ошибка обновления настройки', 'error');
				}
			});
		}
	}
	
	// Добавляем обработчик клика по всему блоку hCaptcha
	const hcaptchaBlock = document.querySelector('label[for="fhcap"]');
	if (hcaptchaBlock) {
		hcaptchaBlock.addEventListener('click', function(e) {
			// Если клик по самому чекбоксу, не обрабатываем (обработает событие change)
			if (e.target.classList.contains('cl_hcap')) {
				return;
			}
			
			// Обрабатываем клик по остальной части блока
			const checkbox = this.querySelector('.cl_hcap');
			if (checkbox && !checkbox.checked) {
				e.preventDefault();
				e.stopPropagation();
				handleHCaptchaClick(e);
			}
		});
		console.log('🚧 Обработчик клика по блоку hCaptcha добавлен');
	}
	
	// Добавляем обработчик клика по всему блоку AWS
	const awsBlock = document.querySelector('label[for="faws"]');
	if (awsBlock) {
		awsBlock.addEventListener('click', function(e) {
			// Если клик по самому чекбоксу, не обрабатываем (обработает событие change)
			if (e.target.classList.contains('cl_aws')) {
				return;
			}
			
			// Обрабатываем клик по остальной части блока
			const checkbox = this.querySelector('.cl_aws');
			if (checkbox && !checkbox.checked) {
				e.preventDefault();
				e.stopPropagation();
				handleAWSClick(e);
			}
		});
		console.log('🚧 Обработчик клика по блоку AWS добавлен');
	}
	
	// Настраиваем кнопки конфигурации
	setupConfigButtons();
	
	// Проверяем наличие кнопки активации в статическом HTML и добавляем обработчик
	const staticActivateBtn = document.getElementById('activateButton');
	if (staticActivateBtn && !staticActivateBtn.hasAttribute('data-listener-added')) {
		staticActivateBtn.addEventListener('click', function(e) {
			e.preventDefault();
			e.stopPropagation();
			console.log('🖱️ Клик по статической кнопке активации');
			activatePlugin();
		});
		staticActivateBtn.setAttribute('data-listener-added', 'true');
		console.log('✅ Обработчик для статической кнопки активации добавлен');
	}
	
	console.log('✅ Все обработчики событий настроены');
}

// Функция debounce для ограничения частоты вызовов
function debounce(func, wait) {
	let timeout;
	return function executedFunction(...args) {
		const later = () => {
			clearTimeout(timeout);
			func(...args);
		};
		clearTimeout(timeout);
		timeout = setTimeout(later, wait);
	};
}

// Обновление данных в фоне без блокировки UI
async function refreshDataInBackground() {
	try {
		const res = await sendChrome.X('get', {});
		if (res && res.received && isDataValid(res.received)) {
			// Проверяем, изменились ли данные значительно
			const hasSignificantChanges = !cachedData || 
				JSON.stringify(cachedData.received) !== JSON.stringify(res.received);
			
			if (hasSignificantChanges) {
				cachedData = res;
				populateUI(res.received);
				console.log('Данные обновлены в фоне');
			}
		} else {
			console.warn('Фоновое обновление: получены некорректные данные');
		}
	} catch (error) {
		console.error('Ошибка фонового обновления данных:', error);
	}
}

delay = ms => new Promise(res => setTimeout(res, ms));

// Функция для загрузки внешней конфигурации
async function loadExternalConfig() {
	if (!configManagerReady || !window.configManager) {
		console.warn('ConfigManager не готов для загрузки конфигурации');
		return false;
	}
	
	try {
		const result = await window.configManager.reloadConfig();
		if (result && result.success) {
			console.log('Конфигурация успешно загружена');
			return true;
		} else {
			console.warn('Частичная загрузка конфигурации');
			return false;
		}
	} catch (error) {
		console.error('Ошибка загрузки конфигурации:', error);
		return false;
	}
}

// Быстрая загрузка из кэша
async function loadFromCache() {
	if (!cachedData || !cachedData.received) {
		console.warn('Нет данных в кэше для загрузки');
		return false;
	}
	
	try {
		// Проверяем валидность данных перед использованием
		if (!isDataValid(cachedData.received)) {
			console.warn('Данные в кэше некорректны, перезагружаем');
			return await retryDataLoad();
		}
		
		populateUI(cachedData.received);
		console.log('UI заполнен из кэша');
		return true;
	} catch (error) {
		console.error('Ошибка загрузки из кэша:', error);
		return await retryDataLoad();
	}
}

// Обновление данных с сервера
async function refreshData() {
	try {
		const res = await sendChrome.X('get', {});
		if (res && res.received) {
			cachedData = res;
			populateUI(res.received);
			console.log('Данные обновлены с сервера');
			return true;
		} else {
			console.warn('Не удалось получить данные с сервера');
			return false;
		}
	} catch (error) {
		console.error('Ошибка обновления данных:', error);
		return false;
	}
}

// Функция для заполнения UI данными
function populateUI(set) {
	if (!set) {
		console.warn('Нет данных для заполнения UI');
		return;
	}
	
	try {
		// Добавлен 'geetest' в массив serviceTypes
		const serviceTypes = ['tiktok','oth','hcap','recap','geetest','funcap','binance','aws'];
		
		// API ключ
		const apiKeyInput = document.querySelector(".block_key_input");
		if (apiKeyInput) { 
			const apiKey = set.apikey || '';
			if (apiKeyInput.value !== apiKey) {
				apiKeyInput.value = apiKey;
			}
			updateApiKeyIndicator(apiKey);
		}
		
		// Баланс - обновляем с учетом переводов
		const balanceEl = document.querySelector(".block_key_header_name span");
		if (balanceEl) { 
			const currentBalance = set.balans || '0';
			balanceEl.innerText = currentBalance;
			
			// Обновляем весь текст баланса с учетом языка
			const lang = getBrowserLanguage();
			const t = translations[lang];
			const balanceContainer = document.querySelector('.block_key_header_name');
			if (balanceContainer) {
				balanceContainer.innerHTML = `${t.headerBalance} <span class="balans">${currentBalance}</span>`;
			}
		}
		
		// Главный переключатель
		let mainOff = false;
		const mainToggle = document.querySelector(".cl_onoff");
		if (mainToggle) { 
			const isOn = set.onoff === '1' || set.onoff === 1;
			if (mainToggle.checked !== isOn) {
				mainToggle.checked = isOn;
			}
			mainOff = isOn;
		}
		
		// Скрытие/показ блоков в зависимости от главного переключателя
		const hiddenBlocks = document.querySelectorAll(".not_show_block");
		if (hiddenBlocks && hiddenBlocks.length > 0) { 
			hiddenBlocks.forEach(block => {
				const shouldHide = mainOff;
				if (block.style.display !== (shouldHide ? 'none' : 'block')) {
					block.style.display = shouldHide ? 'none' : 'block';
				}
			});
		}
		
		// Сообщение об ошибке
		const errorBlock = document.querySelector(".block_error");
		if (errorBlock && 'onoff2' in set) { 
			const lang = getBrowserLanguage();
			const errorMessage = lang === 'ru' 
				? 'Возможно не рабочий ключ, или пустой баланс' 
				: 'Possibly invalid key or empty balance';
			const newErrorText = (set.onoff2 === '0' || set.onoff2 === 0) ? errorMessage : '';
			if (errorBlock.innerText !== newErrorText) {
				errorBlock.innerText = newErrorText;
			}
		}
		
		// Настройки сервисов
		serviceTypes.forEach(serviceType => {
			const checkbox = document.querySelector(".cl_" + serviceType);
			if (checkbox) {
				let shouldBeChecked = false;
				
				if (set && serviceType in set && mainOff) { 
					// Специальная обработка для hCaptcha и AWS - всегда выключаем
					if (serviceType === 'hcap' || serviceType === 'aws') {
						shouldBeChecked = false;
					} else {
						shouldBeChecked = set[serviceType] === '1' || set[serviceType] === 1;
					}
				}
				
				if (checkbox.checked !== shouldBeChecked) {
					checkbox.checked = shouldBeChecked;
				}
			}
		});
		
		// Проверяем состояние плагина после заполнения UI с небольшой задержкой
		setTimeout(() => {
			checkPluginState();
		}, 50);
		
		console.log('UI успешно заполнен данными');
		
	} catch (error) {
		console.error('Ошибка заполнения UI:', error);
	}
}

// Настройка обработчиков кнопок
function setupConfigButtons() {
	const reloadBtn = document.getElementById('reloadConfigBtn');
	const statusBtn = document.getElementById('configStatusBtn');
	
	if (reloadBtn) {
		reloadBtn.addEventListener('click', async function() {
			this.disabled = true;
			this.textContent = '⏳ Loading...';
			
			try {
				await loadExternalConfig();
				await refreshData();
				await updateConfigStatus();
				showNotification('Конфигурация перезагружена', 'success');
			} catch (error) {
				console.error('Ошибка перезагрузки:', error);
				showNotification('Ошибка перезагрузки', 'error');
			} finally {
				this.disabled = false;
				this.textContent = '🔄 Reload';
			}
		});
	}
	
	if (statusBtn) {
		statusBtn.addEventListener('click', async function() {
			await showConfigStatus();
		});
	}
}

// Обновление статуса конфигурации
async function updateConfigStatus() {
	if (!configManagerReady || !window.configManager) return;
	
	try {
		const status = await window.configManager.getConfigStatus();
		const statusEl = document.getElementById('configStatus');
		
		if (statusEl) {
			statusEl.className = 'config-status ' + (status.inSync ? 'synced' : 'modified');
			statusEl.title = status.inSync 
				? 'Настройки синхронизированы с файлом'
				: `${status.differences} настроек отличаются от файла`;
		}
	} catch (error) {
		console.error('Ошибка обновления статуса:', error);
		const statusEl = document.getElementById('configStatus');
		if (statusEl) {
			statusEl.className = 'config-status error';
			statusEl.title = 'Ошибка проверки статуса';
		}
	}
}

// Показ детального статуса конфигурации
async function showConfigStatus() {
	if (!configManagerReady || !window.configManager) {
		showNotification('ConfigManager не доступен', 'error');
		return;
	}
	
	try {
		const status = await window.configManager.getConfigStatus();
		const differences = status.details;
		
		let message = status.inSync 
			? '✅ Все настройки синхронизированы с config.json'
			: `⚠️ Найдено ${status.differences} различий:\n\n`;
		
		if (!status.inSync) {
			if (differences.apikey && differences.apikey.differs) {
				message += `🔑 API Key:\n  Файл: "${differences.apikey.file}"\n  Текущий: "${differences.apikey.current}"\n\n`;
			}
			
			if (differences.settings) {
				for (let key in differences.settings) {
					const setting = differences.settings[key];
					if (setting.differs) {
						message += `⚙️ ${key}: файл="${setting.file}", текущий="${setting.current}"\n`;
					}
				}
			}
		}
		
		showNotification(message, status.inSync ? 'success' : 'warning', 5000);
	} catch (error) {
		console.error('Ошибка получения статуса:', error);
		showNotification('Ошибка получения статуса', 'error');
	}
}

// Упрощенная функция load для совместимости
async function load(){
	return await refreshData();
}

// Функция для обновления индикатора источника API ключа
async function updateApiKeyIndicator(apikey) {
	// Удаляем существующий индикатор
	const existingIndicator = document.querySelector('.config-source-indicator');
	if (existingIndicator) {
		existingIndicator.remove();
	}
	
	// Создаем новый индикатор, если есть API ключ
	if (apikey && apikey.trim() !== '') {
		const indicator = document.createElement('div');
		indicator.className = 'config-source-indicator';
		
		// Проверяем источник ключа
		if (configManagerReady && window.configManager) {
			try {
				const config = await window.configManager.loadConfig();
				if (config && config.apikey === apikey) {
					indicator.textContent = '📁 Loaded from config.json';
					indicator.style.color = '#4CAF50';
				} else {
					indicator.textContent = '🔧 Set manually';
					indicator.style.color = '#2196F3';
				}
			} catch (error) {
				console.warn('Не удалось проверить источник API ключа:', error);
				indicator.textContent = '🔧 Set manually';
				indicator.style.color = '#2196F3';
			}
		} else {
			indicator.textContent = '🔧 Set manually';
			indicator.style.color = '#2196F3';
		}
		
		const keyBlock = document.querySelector('.block_key_text');
		if (keyBlock) {
			keyBlock.appendChild(indicator);
		}
	}
}

// Показ/скрытие индикатора загрузки
function showLoadingIndicator(show) {
	let indicator = document.getElementById('loadingIndicator');
	
	if (show && !indicator) {
		indicator = document.createElement('div');
		indicator.id = 'loadingIndicator';
		indicator.className = 'loading-indicator';
		indicator.innerHTML = '<span>⏳ Загрузка данных...</span>';
		indicator.style.cssText = `
			position: fixed;
			top: 50%;
			left: 50%;
			transform: translate(-50%, -50%);
			background: rgba(0, 0, 0, 0.8);
			color: white;
			padding: 12px 20px;
			border-radius: 8px;
			z-index: 10001;
			font-size: 12px;
		`;
		document.body.appendChild(indicator);
	} else if (!show && indicator) {
		indicator.remove();
	}
}

class sendChrome {
    static async X() {
        return new Promise(resolve => {
            try {
                chrome.runtime.sendMessage([...arguments], response => {
					if (chrome.runtime.lastError) {
						console.error('Chrome runtime error:', chrome.runtime.lastError);
						resolve(null);
					} else {
						resolve(response);
					}
				});
            } catch (e) {
                console.error('sendChrome.X failed:', e);
                resolve(null);
            }
        });
    }
	
	static async fetch (url, options) {
        return await this.X('sendChrome.fetch', {url, options});
    }
	
	static inject_file(file) {
        return new Promise(resolve => {
            const $script = document.createElement('script');
            $script.src = chrome.runtime.getURL(file);
            $script.onload = resolve;
            $script.onerror = () => {
				console.error('Не удалось загрузить файл:', file);
				resolve();
			};
            (document.head || document.documentElement).appendChild($script);
        });
    }
}

const getStorageData = key =>
	new Promise((resolve, reject) =>
    chrome.storage.sync.get(key, result =>
      chrome.runtime.lastError
        ? reject(Error(chrome.runtime.lastError.message))
        : resolve(result)
    )
  )

const setStorageData = function(name, val){
	let d = {};
	d[name] = val;
	return new Promise((resolve, reject) => {
		chrome.storage.sync.set(d, () =>
		  chrome.runtime.lastError
			? reject(Error(chrome.runtime.lastError.message))
			: resolve()
		)
	})
}

// Функция для показа уведомлений
function showNotification(message, type = 'info', duration = 3000) {
	const notification = document.createElement('div');
	notification.className = `notification ${type}`;
	notification.style.cssText = `
		position: fixed;
		top: 10px;
		right: 10px;
		padding: 10px 15px;
		border-radius: 4px;
		color: white;
		font-size: 14px;
		z-index: 10000;
		max-width: 300px;
		word-wrap: break-word;
		transition: opacity 0.3s ease;
		opacity: 0;
	`;
	
	// Цвета для разных типов уведомлений
	const colors = {
		'success': '#4CAF50',
		'error': '#f44336',
		'warning': '#ff9800',
		'info': '#2196F3'
	};
	
	notification.style.backgroundColor = colors[type] || colors.info;
	
	// Поддержка многострочных сообщений
	if (message.includes('\n')) {
		notification.innerHTML = message.replace(/\n/g, '<br>');
	} else {
		notification.textContent = message;
	}
	
	document.body.appendChild(notification);
	
	// Анимация появления
	setTimeout(() => notification.style.opacity = '1', 10);
	
	// Удаление через указанное время
	setTimeout(() => {
		notification.style.opacity = '0';
		setTimeout(() => {
			if (notification.parentNode) {
				document.body.removeChild(notification);
			}
		}, 300);
	}, duration);
}

// Запуск предзагрузки при загрузке скрипта
(async function initializePreload() {
	console.log('Инициализация предзагрузки...');
	
	// Запускаем предзагрузку немедленно
	if (document.readyState === 'loading') {
		// Если DOM еще загружается, ждем его загрузки
		document.addEventListener('DOMContentLoaded', () => {
			console.log('DOM загружен, запускаем предзагрузку');
			preloadData();
		});
	} else {
		// DOM уже загружен, запускаем предзагрузку сразу
		console.log('DOM уже загружен, запускаем предзагрузку немедленно');
		preloadData();
	}
})();

// Добавляем обработчик для кнопки активации, если она уже существует в DOM
document.addEventListener('DOMContentLoaded', function() {
	// Дополнительная проверка для существующей кнопки в HTML
	const existingActivateBtn = document.getElementById('activateButton');
	if (existingActivateBtn && !existingActivateBtn.hasAttribute('data-listener-added')) {
		existingActivateBtn.addEventListener('click', function(e) {
			e.preventDefault();
			e.stopPropagation();
			console.log('🖱️ Клик по существующей кнопке активации');
			activatePlugin();
		});
		existingActivateBtn.setAttribute('data-listener-added', 'true');
		console.log('✅ Обработчик для существующей кнопки активации добавлен');
	}
});