class ConfigManager {
    constructor() {
        this.configPath = 'config.json';
        this.configExists = false;
        this.defaultConfig = {
            apikey: '',
            settings: {
                onoff: '1',
                tiktok: '0',
                recap: '1', 
                oth: '0',
                binance: '0',
                funcap: '0',
                hcap: '0',
                geetest: '0',
                aws: '0'
            }
        };
    }

    async loadConfig() {
        try {
            const response = await fetch(chrome.runtime.getURL(this.configPath));
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const config = await response.json();
            this.configExists = true;
            return this.validateConfig(config);
        } catch (error) {
            console.warn('Не удалось загрузить config.json:', error.message);
            console.info('Файл конфигурации отсутствует - используется ручной режим');
            this.configExists = false;
            return null;
        }
    }

    async checkConfigExists() {
        try {
            const response = await fetch(chrome.runtime.getURL(this.configPath));
            this.configExists = response.ok;
            return this.configExists;
        } catch (error) {
            this.configExists = false;
            return false;
        }
    }

    validateConfig(config) {
        const validatedConfig = { ...this.defaultConfig };
        
        if (config.apikey && typeof config.apikey === 'string') {
            validatedConfig.apikey = config.apikey.trim();
        }
        
        if (config.settings && typeof config.settings === 'object') {
            for (let key in validatedConfig.settings) {
                if (config.settings.hasOwnProperty(key)) {
                    const value = config.settings[key];
                    if (value === '0' || value === '1' || value === 0 || value === 1) {
                        validatedConfig.settings[key] = String(value);
                    }
                }
            }
        }
        
        return validatedConfig;
    }

    async applyConfig(config) {
        if (!this.configExists || !config) {
            console.log('Конфигурация не применена - файл отсутствует или ручной режим');
            return [];
        }

        const results = [];
        
        try {
            if (config.apikey && config.apikey.trim() !== '') {
                const result = await sendChrome.X('set', {
                    "n": "apikey", 
                    "v": config.apikey
                });
                results.push({ key: 'apikey', success: !!result });
            }
            
            if (config.settings) {
                for (let setting in config.settings) {
                    const result = await sendChrome.X('set', {
                        "n": setting,
                        "v": config.settings[setting]
                    });
                    results.push({ key: setting, success: !!result });
                }
            }
            
            return results;
        } catch (error) {
            console.error('Ошибка при применении конфигурации:', error);
            return [];
        }
    }

    async getCurrentSettings() {
        try {
            const response = await sendChrome.X('get', {});
            return response && response.received ? response.received : {};
        } catch (error) {
            console.error('Ошибка при получении текущих настроек:', error);
            return {};
        }
    }

    async compareConfigs() {
        const fileConfig = await this.loadConfig();
        const currentSettings = await this.getCurrentSettings();
        
        if (!this.configExists || !fileConfig) {
            return {
                manualMode: true,
                message: 'Ручной режим - файл config.json отсутствует'
            };
        }
        
        const differences = {
            manualMode: false,
            apikey: {
                file: fileConfig.apikey || '',
                current: currentSettings.apikey || '',
                differs: (fileConfig.apikey || '') !== (currentSettings.apikey || '')
            },
            settings: {}
        };
        
        for (let key in fileConfig.settings) {
            differences.settings[key] = {
                file: fileConfig.settings[key],
                current: currentSettings[key] || '0',
                differs: fileConfig.settings[key] !== (currentSettings[key] || '0')
            };
        }
        
        return differences;
    }

    async reloadConfig() {
        console.log('🔄 Перезагрузка конфигурации...');
        
        await this.checkConfigExists();
        
        if (!this.configExists) {
            console.log('📄 Файл config.json отсутствует - работаем в ручном режиме');
            return {
                config: null,
                results: [],
                success: true,
                manualMode: true,
                message: 'Ручной режим активен'
            };
        }
        
        const config = await this.loadConfig();
        const results = await this.applyConfig(config);
        
        const successCount = results.filter(r => r.success).length;
        const totalCount = results.length;
        
        console.log(`✅ Применено ${successCount}/${totalCount} настроек`);
        
        return {
            config,
            results,
            success: successCount === totalCount,
            manualMode: false
        };
    }

    async createBackup() {
        const currentSettings = await this.getCurrentSettings();
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
        
        const backup = {
            timestamp,
            apikey: currentSettings.apikey || '',
            settings: {}
        };
        
        const settingsKeys = ['onoff', 'tiktok', 'recap', 'oth', 'binance', 'funcap', 'hcap', 'geetest', 'aws'];
        settingsKeys.forEach(key => {
            backup.settings[key] = currentSettings[key] || '0';
        });
        
        return backup;
    }

    async getConfigStatus() {
        const differences = await this.compareConfigs();
        
        if (differences.manualMode) {
            return {
                inSync: true,
                differences: 0,
                manualMode: true,
                message: differences.message,
                details: differences
            };
        }
        
        const hasApiKeyDiff = differences.apikey.differs;
        const settingsDiffs = Object.values(differences.settings).filter(s => s.differs);
        
        return {
            inSync: !hasApiKeyDiff && settingsDiffs.length === 0,
            differences: settingsDiffs.length + (hasApiKeyDiff ? 1 : 0),
            manualMode: false,
            details: differences
        };
    }

    isManualMode() {
        return !this.configExists;
    }

    enableManualMode() {
        this.configExists = false;
        console.log('🔧 Ручной режим принудительно включен');
    }
}

window.configManager = new ConfigManager();