// Проверяем доступность Firefox API
if (typeof browser === 'undefined') {
    console.error('❌ Firefox browser API недоступен!');
    document.addEventListener('DOMContentLoaded', () => {
        document.body.innerHTML = `
            <div style="padding: 20px; text-align: center; color: red;">
                <h3>Ошибка загрузки</h3>
                <p>Firefox API недоступен. Пожалуйста, перезагрузите расширение.</p>
            </div>
        `;
    });
} else {
    console.log('✅ Firefox browser API доступен');
}

// Многоязычные тексты
const translations = {
    en: {
        title: "Plugin Disabled",
        message: "Please enable the plugin to access captcha solving features.",
        button: "Enable Plugin",
        headerBalance: "Your balance:",
        topUp: "Top up",
        apiPlaceholder: "Enter your API KEY",
        reloadBtn: "🔄 Reload",
        statusBtn: "📊 Status",
        captchaTypes: "Type of captchas:",
        contact: "Have a question?",
        pluginOn: "Plugin is ON!",
        hcaptchaInDevelopment: "🚧 In Development\n\nhCaptcha support is currently being developed.\nThis feature will be available in a future update.",
        awsInDevelopment: "🚧 In Development\n\nAWS Captcha support is currently being developed.\nThis feature will be available in a future update."
    },
    ru: {
        title: "Плагин отключен",
        message: "Пожалуйста, включите плагин для доступа к функциям решения капчи.",
        button: "Включить плагин",
        headerBalance: "Ваш баланс:",
        topUp: "Пополнить",
        apiPlaceholder: "Введите ваш API ключ",
        reloadBtn: "🔄 Перезагрузить",
        statusBtn: "📊 Статус",
        captchaTypes: "Типы капч:",
        contact: "Есть вопрос?",
        pluginOn: "Плагин включен!",
        hcaptchaInDevelopment: "🚧 В разработке\n\nПоддержка hCaptcha сейчас разрабатывается.\nЭта функция будет доступна в будущих обновлениях.",
        awsInDevelopment: "🚧 В разработке\n\nПоддержка AWS Captcha сейчас разрабатывается.\nЭта функция будет доступна в будущих обновлениях."
    }
};

// Определение языка браузера для Firefox
function getBrowserLanguage() {
    try {
        if (typeof browser !== 'undefined' && browser.i18n && browser.i18n.getUILanguage) {
            const lang = browser.i18n.getUILanguage() || navigator.language || navigator.languages[0] || 'en';
            const langCode = lang.toLowerCase().split('-')[0];
            return langCode === 'ru' ? 'ru' : 'en';
        } else {
            const lang = navigator.language || navigator.languages[0] || 'en';
            const langCode = lang.toLowerCase().split('-')[0];
            return langCode === 'ru' ? 'ru' : 'en';
        }
    } catch (error) {
        console.warn('Ошибка определения языка:', error);
        return 'en';
    }
}

// Firefox-совместимый класс sendChrome
class sendChrome {
    static async X(...args) {
        return new Promise((resolve) => {
            try {
                console.log('📤 Отправка сообщения в background:', args);
                
                if (typeof browser === 'undefined' || !browser.runtime || !browser.runtime.sendMessage) {
                    console.error('❌ browser.runtime.sendMessage недоступен');
                    resolve(null);
                    return;
                }

                if (!Array.isArray(args)) {
                    console.error('❌ Аргументы должны быть массивом');
                    resolve(null);
                    return;
                }

                // Firefox возвращает Promise
                browser.runtime.sendMessage(args)
                    .then(response => {
                        console.log('📥 Получен ответ от background:', response);
                        resolve(response);
                    })
                    .catch(error => {
                        console.error('❌ Ошибка sendMessage:', error);
                        resolve(null);
                    });
            } catch (e) {
                console.error('❌ Исключение в sendMessage:', e);
                resolve(null);
            }
        });
    }
}

// Функция для обработки клика по hCaptcha
function handleHCaptchaClick(event) {
    event.preventDefault();
    event.stopPropagation();
    
    const lang = getBrowserLanguage();
    const t = translations[lang];
    
    showNotification(t.hcaptchaInDevelopment, 'warning', 5000);
    
    const hcaptchaCheckbox = document.querySelector('.cl_hcap');
    if (hcaptchaCheckbox && hcaptchaCheckbox.checked) {
        hcaptchaCheckbox.checked = false;
        console.log('🚧 hCaptcha отключен - функция в разработке');
    }
}

// Функция для обработки клика по AWS Captcha
function handleAWSClick(event) {
    event.preventDefault();
    event.stopPropagation();
    
    const lang = getBrowserLanguage();
    const t = translations[lang];
    
    showNotification(t.awsInDevelopment, 'warning', 5000);
    
    const awsCheckbox = document.querySelector('.cl_aws');
    if (awsCheckbox && awsCheckbox.checked) {
        awsCheckbox.checked = false;
        console.log('🚧 AWS Captcha отключен - функция в разработке');
    }
}

// Применение переводов
function applyTranslations() {
    const lang = getBrowserLanguage();
    const t = translations[lang];
    
    const inactiveTitle = document.getElementById('inactiveTitle');
    const inactiveMessage = document.getElementById('inactiveMessage');
    const activateButton = document.getElementById('activateButton');
    
    if (inactiveTitle) inactiveTitle.textContent = t.title;
    if (inactiveMessage) inactiveMessage.textContent = t.message;
    if (activateButton) activateButton.textContent = t.button;
    
    const balanceEl = document.querySelector('.block_key_header_name');
    if (balanceEl) {
        const currentBalance = balanceEl.querySelector('.balans')?.textContent || '0';
        balanceEl.innerHTML = `${t.headerBalance} <span class="balans">${currentBalance}</span>`;
    }
    
    const topUpLink = document.querySelector('.block_key_header_link a');
    if (topUpLink) topUpLink.textContent = t.topUp;
    
    const apiInput = document.querySelector('.block_key_input');
    if (apiInput) apiInput.placeholder = t.apiPlaceholder;
    
    const reloadBtn = document.getElementById('reloadConfigBtn');
    if (reloadBtn) reloadBtn.title = t.reloadBtn.replace('🔄 ', '');
    
    const statusBtn = document.getElementById('configStatusBtn');
    if (statusBtn) statusBtn.title = t.statusBtn.replace('📊 ', '');
    
    const contactLink = document.querySelector('.repot_contact a');
    if (contactLink) contactLink.textContent = t.contact;
    
    const pluginOnMsg = document.querySelector('.not_show_block');
    if (pluginOnMsg && !pluginOnMsg.querySelector('img')) {
        pluginOnMsg.textContent = t.pluginOn;
    }
}

// Обновляем CSS для мультиязычности
function updateCSSForLanguage() {
    const lang = getBrowserLanguage();
    const t = translations[lang];
    
    const existingStyle = document.getElementById('lang-style');
    if (existingStyle) {
        existingStyle.remove();
    }
    
    const style = document.createElement('style');
    style.id = 'lang-style';
    style.textContent = `
        .auto_ch::before {
            content: "${t.captchaTypes}";
            display: block;
            font-weight: 600;
            color: #374151;
            margin-bottom: 8px;
            font-size: 12px;
        }
    `;
    document.head.appendChild(style);
}

// Функция активации плагина
function activatePlugin() {
    const mainToggle = document.querySelector(".cl_onoff");
    const wrapper = document.getElementById('pluginWrapper') || document.querySelector('.wrapper');
    const overlay = document.getElementById('inactiveOverlay');
    
    if (mainToggle) {
        mainToggle.checked = true;
        
        if (overlay) overlay.classList.add('hidden');
        if (wrapper) wrapper.classList.remove('plugin-inactive');
        
        mainToggle.dispatchEvent(new Event('change'));
        
        console.log('🔄 Плагин активирован через overlay');
    }
}

// Функция проверки состояния плагина
function checkPluginState() {
    const mainToggle = document.querySelector(".cl_onoff");
    const wrapper = document.getElementById('pluginWrapper') || document.querySelector('.wrapper');
    const overlay = document.getElementById('inactiveOverlay');
    
    if (!mainToggle || !overlay || !wrapper) {
        console.warn('⚠️ Элементы для проверки состояния плагина не найдены');
        return;
    }
    
    if (!mainToggle.checked) {
        overlay.classList.remove('hidden');
        wrapper.classList.add('plugin-inactive');
    } else {
        overlay.classList.add('hidden');
        wrapper.classList.remove('plugin-inactive');
    }
}

// Создание overlay элемента если его нет
function createInactiveOverlay() {
    if (document.getElementById('inactiveOverlay')) {
        return;
    }
    
    const wrapper = document.querySelector('.wrapper');
    if (!wrapper) {
        console.warn('⚠️ Wrapper не найден, невозможно создать overlay');
        return;
    }
    
    if (!wrapper.id) {
        wrapper.id = 'pluginWrapper';
    }
    
    const overlay = document.createElement('div');
    overlay.className = 'plugin-inactive-overlay hidden';
    overlay.id = 'inactiveOverlay';
    
    overlay.innerHTML = `
        <div class="inactive-icon"></div>
        <div class="inactive-title" id="inactiveTitle">Plugin Disabled</div>
        <div class="inactive-message" id="inactiveMessage">
            Please enable the plugin to access captcha solving features.
        </div>
        <button class="activate-button" id="activateButton">
            Enable Plugin
        </button>
    `;
    
    wrapper.appendChild(overlay);
    console.log('✅ Inactive overlay создан');
}

// Глобальные переменные для управления данными
let cachedData = null;
let isDataLoading = false;
let preloadPromise = null;
let configManagerReady = false;

// Функция delay
const delay = ms => new Promise(res => setTimeout(res, ms));

// Загрузка скрипта ConfigManager для Firefox
async function loadConfigManagerScript() {
    return new Promise((resolve, reject) => {
        try {
            const script = document.createElement('script');
            script.src = browser.runtime.getURL('web/js/all/config-manager.js');
            script.onload = () => {
                console.log('✅ ConfigManager загружен');
                resolve();
            };
            script.onerror = (error) => {
                console.error('❌ Ошибка загрузки ConfigManager:', error);
                reject(error);
            };
            (document.head || document.documentElement).appendChild(script);
        } catch (error) {
            console.error('❌ Исключение при загрузке ConfigManager:', error);
            reject(error);
        }
    });
}

// Ожидание загрузки ConfigManager
async function waitForConfigManager() {
    let attempts = 0;
    const maxAttempts = 50;
    
    while (!window.configManager && attempts < maxAttempts) {
        await delay(50);
        attempts++;
        
        if (attempts === 10 && !window.configManager) {
            try {
                await loadConfigManagerScript();
            } catch (error) {
                console.warn('⚠️ Не удалось загрузить ConfigManager:', error);
            }
        }
    }
    
    configManagerReady = !!window.configManager;
    
    if (!configManagerReady) {
        console.error('❌ ConfigManager не загружен после', maxAttempts, 'попыток');
    } else {
        console.log('✅ ConfigManager готов');
    }
    
    return configManagerReady;
}

// Функция для загрузки внешней конфигурации
async function loadExternalConfig() {
    if (!configManagerReady || !window.configManager) {
        console.warn('⚠️ ConfigManager не готов для загрузки конфигурации');
        return false;
    }
    
    try {
        const result = await window.configManager.reloadConfig();
        if (result && result.success) {
            console.log('✅ Конфигурация успешно загружена');
            return true;
        } else {
            console.warn('⚠️ Частичная загрузка конфигурации');
            return false;
        }
    } catch (error) {
        console.error('❌ Ошибка загрузки конфигурации:', error);
        return false;
    }
}

// Основная функция предзагрузки данных (упрощенная версия)
async function preloadData() {
    if (isDataLoading && preloadPromise) {
        return preloadPromise;
    }
    
    isDataLoading = true;
    
    preloadPromise = (async () => {
        try {
            console.log('🚀 Начинаем предзагрузку данных...');
            
            // Проверяем связь с background script
            console.log('📡 Запрашиваем данные из background script...');
            const res = await sendChrome.X('get', {});
            
            if (res && res.received) {
                cachedData = res;
                console.log('✅ Данные предзагружены:', cachedData);
            } else {
                console.warn('⚠️ Не удалось получить данные для предзагрузки');
                // Попробуем альтернативный способ
                console.log('🔄 Пробуем запросить предзагруженные данные...');
                const preloadedRes = await sendChrome.X('getPreloadedData');
                if (preloadedRes && preloadedRes.received) {
                    cachedData = preloadedRes;
                    console.log('✅ Получены предзагруженные данные:', cachedData);
                }
            }
            
            return cachedData;
            
        } catch (error) {
            console.error('❌ Ошибка предзагрузки данных:', error);
            return null;
        } finally {
            isDataLoading = false;
        }
    })();
    
    return preloadPromise;
}

// Быстрая загрузка из кэша
async function loadFromCache() {
    if (!cachedData || !cachedData.received) {
        console.warn('⚠️ Нет данных в кэше для загрузки');
        return false;
    }
    
    try {
        populateUI(cachedData.received);
        console.log('✅ UI заполнен из кэша');
        return true;
    } catch (error) {
        console.error('❌ Ошибка загрузки из кэша:', error);
        return false;
    }
}

// Обновление данных с сервера
async function refreshData() {
    try {
        console.log('📡 Обновляем данные с сервера...');
        const res = await sendChrome.X('get', {});
        if (res && res.received) {
            cachedData = res;
            populateUI(res.received);
            console.log('✅ Данные обновлены с сервера');
            return true;
        } else {
            console.warn('⚠️ Не удалось получить данные с сервера');
            return false;
        }
    } catch (error) {
        console.error('❌ Ошибка обновления данных:', error);
        return false;
    }
}

// Функция для заполнения UI данными
function populateUI(set) {
    if (!set) {
        console.warn('⚠️ Нет данных для заполнения UI');
        return;
    }
    
    try {
        console.log('🎨 Заполняем UI данными:', set);
        // Добавлен 'geetest' в массив serviceTypes
        const serviceTypes = ['tiktok','oth','hcap','recap','geetest','funcap','binance','aws'];
        
        // API ключ
        const apiKeyInput = document.querySelector(".block_key_input");
        if (apiKeyInput && 'apikey' in set) { 
            apiKeyInput.value = set.apikey || ''; 
        }
        
        // Баланс - обновляем с учетом переводов
        const balanceEl = document.querySelector(".balans");
        if (balanceEl && 'balans' in set) { 
            const currentBalance = set.balans || '0';
            balanceEl.innerText = currentBalance;
            
            const lang = getBrowserLanguage();
            const t = translations[lang];
            const balanceContainer = document.querySelector('.block_key_header_name');
            if (balanceContainer) {
                balanceContainer.innerHTML = `${t.headerBalance} <span class="balans">${currentBalance}</span>`;
            }
        }
        
        // Главный переключатель
        let mainOff = false;
        const mainToggle = document.querySelector(".cl_onoff");
        if (mainToggle && 'onoff' in set) { 
            mainToggle.checked = set['onoff'] == '1'; 
            mainOff = set['onoff'] == '1'; 
        }
        
        // Скрытие/показ блоков в зависимости от главного переключателя
        const hiddenBlocks = document.querySelectorAll(".not_show_block");
        if (hiddenBlocks && hiddenBlocks.length > 0) { 
            hiddenBlocks.forEach(block => {
                block.style.display = mainOff ? 'none' : 'block';
            });
        }
        
        // Сообщение об ошибке
        const errorBlock = document.querySelector(".block_error");
        if (errorBlock && 'onoff2' in set) { 
            const lang = getBrowserLanguage();
            const errorMessage = lang === 'ru' 
                ? 'Возможно не рабочий ключ, или пустой баланс' 
                : 'Possibly invalid key or empty balance';
            errorBlock.innerText = set['onoff2'] == '0' ? errorMessage : '';
        }
        
        // Настройки сервисов
        serviceTypes.forEach(serviceType => {
            const checkbox = document.querySelector(".cl_" + serviceType);
            if (checkbox) {
                if (set && serviceType in set && mainOff) { 
                    // Специальная обработка для hCaptcha и AWS - всегда выключаем
                    if (serviceType === 'hcap' || serviceType === 'aws') {
                        checkbox.checked = false;
                    } else {
                        checkbox.checked = set[serviceType] == '1';
                    }
                } else { 
                    checkbox.checked = false; 
                }
            }
        });
        
        // Проверяем состояние плагина после заполнения UI
        checkPluginState();
        
        // Обновляем индикаторы активных блоков
        updateActiveBlocks();
        
        console.log('✅ UI успешно заполнен данными');
        
    } catch (error) {
        console.error('❌ Ошибка заполнения UI:', error);
    }
}

// Функция обновления активных блоков
function updateActiveBlocks() {
    const switchBtns = document.querySelectorAll('.switchBtn');
    switchBtns.forEach(btn => {
        const checkbox = btn.querySelector('input[type="checkbox"]');
        if (checkbox) {
            if (checkbox.checked) {
                btn.classList.add('active');
            } else {
                btn.classList.remove('active');
            }
        }
    });
}

// Настройка всех обработчиков событий
function setupEventHandlers() {
    console.log('🔧 Настраиваем обработчики событий...');
    
    // Обработчик API ключа
    const sourceEl = document.querySelector(".block_key_input");
    
    const inputHandler = async function(e) {
        const apiKey = e.target.value;
        try {
            console.log('🔑 Обновляем API ключ...');
            const res = await sendChrome.X('set', {"n":"apikey", 'v': apiKey});
            await delay(200);
            console.log('✅ API key updated:', res);
            
            await load();
            
            cachedData = await sendChrome.X('get', {});
        } catch (error) {
            console.error('❌ Ошибка обновления API ключа:', error);
            showNotification('Ошибка обновления API ключа', 'error');
        }
    };
    
    if (sourceEl) {
        sourceEl.addEventListener('input', debounce(inputHandler, 500));
        sourceEl.addEventListener('propertychange', debounce(inputHandler, 500));
        console.log('✅ Обработчики API ключа добавлены');
    }
    
    // Обработчики чекбоксов (добавлен .cl_geetest и .cl_aws)
    const checkboxes = document.querySelectorAll('.cl_onoff,.cl_tiktok,.cl_oth,.cl_hcap,.cl_recap,.cl_geetest,.cl_funcap,.cl_binance,.cl_aws');
    console.log('🔍 Найдено чекбоксов:', checkboxes.length);
    
    for (let i = 0; i < checkboxes.length; i++) {
        const checkbox = checkboxes[i];
        console.log(`📋 Добавляем обработчик для: ${checkbox.name || checkbox.className}`);
        
        // Специальная обработка для hCaptcha
        if (checkbox.classList.contains('cl_hcap')) {
            checkbox.addEventListener("change", function(e) {
                if (this.checked) {
                    handleHCaptchaClick(e);
                }
            });
            console.log('🚧 Специальный обработчик для hCaptcha добавлен');
        }
        // Специальная обработка для AWS Captcha
        else if (checkbox.classList.contains('cl_aws')) {
            checkbox.addEventListener("change", function(e) {
                if (this.checked) {
                    handleAWSClick(e);
                }
            });
            console.log('🚧 Специальный обработчик для AWS Captcha добавлен');
        } else {
            // Стандартная обработка для остальных чекбоксов
            checkbox.addEventListener("change", async function(e) {
                const name = this.name;
                const value = this.checked ? '1' : '';
                
                console.log(`🔄 Изменение настройки: ${name} = ${value}`);
                
                try {
                    await sendChrome.X('set', {"n": name, 'v': value});
                    await delay(200);
                    await load();
                    
                    if (name === 'onoff') {
                        console.log('🔄 Главный переключатель изменен, проверяем состояние...');
                        checkPluginState();
                    }
                    
                    // Обновляем визуальные индикаторы
                    updateActiveBlocks();
                    
                    cachedData = await sendChrome.X('get', {});
                } catch (error) {
                    console.error('❌ Ошибка обновления настройки:', error);
                    showNotification('Ошибка обновления настройки', 'error');
                }
            });
        }
    }
    
    // Добавляем обработчик клика по всему блоку hCaptcha
    const hcaptchaBlock = document.querySelector('label[for="fhcap"]');
    if (hcaptchaBlock) {
        hcaptchaBlock.addEventListener('click', function(e) {
            if (e.target.classList.contains('cl_hcap')) {
                return;
            }
            
            const checkbox = this.querySelector('.cl_hcap');
            if (checkbox && !checkbox.checked) {
                e.preventDefault();
                e.stopPropagation();
                handleHCaptchaClick(e);
            }
        });
        console.log('🚧 Обработчик клика по блоку hCaptcha добавлен');
    }
    
    // Добавляем обработчик клика по всему блоку AWS
    const awsBlock = document.querySelector('label[for="faws"]');
    if (awsBlock) {
        awsBlock.addEventListener('click', function(e) {
            if (e.target.classList.contains('cl_aws')) {
                return;
            }
            
            const checkbox = this.querySelector('.cl_aws');
            if (checkbox && !checkbox.checked) {
                e.preventDefault();
                e.stopPropagation();
                handleAWSClick(e);
            }
        });
        console.log('🚧 Обработчик клика по блоку AWS добавлен');
    }
    
    // Обработчик кнопки активации
    const activateButton = document.getElementById('activateButton');
    if (activateButton) {
        activateButton.addEventListener('click', function(e) {
            e.preventDefault();
            e.stopPropagation();
            console.log('🖱️ Клик по кнопке активации');
            activatePlugin();
        });
        console.log('✅ Обработчик кнопки активации добавлен');
    }
    
    setupConfigButtons();
    
    console.log('✅ Все обработчики событий настроены');
}

// Функция debounce для ограничения частоты вызовов
function debounce(func, wait) {
    let timeout;
    return function executedFunction(...args) {
        const later = () => {
            clearTimeout(timeout);
            func(...args);
        };
        clearTimeout(timeout);
        timeout = setTimeout(later, wait);
    };
}

// Настройка обработчиков кнопок
function setupConfigButtons() {
    const reloadBtn = document.getElementById('reloadConfigBtn');
    const statusBtn = document.getElementById('configStatusBtn');
    
    if (reloadBtn) {
        reloadBtn.addEventListener('click', async function() {
            this.disabled = true;
            this.textContent = '⏳ Loading...';
            
            try {
                await loadExternalConfig();
                await refreshData();
                showNotification('Конфигурация перезагружена', 'success');
            } catch (error) {
                console.error('❌ Ошибка перезагрузки:', error);
                showNotification('Ошибка перезагрузки', 'error');
            } finally {
                this.disabled = false;
                this.textContent = '🔄 Reload';
            }
        });
    }
    
    if (statusBtn) {
        statusBtn.addEventListener('click', async function() {
            await showConfigStatus();
        });
    }
}

// Обновление статуса конфигурации
async function updateConfigStatus() {
    if (!configManagerReady || !window.configManager) return;
    
    try {
        const status = await window.configManager.getConfigStatus();
        const statusEl = document.getElementById('configStatus');
        
        if (statusEl) {
            statusEl.className = 'config-status ' + (status.inSync ? 'synced' : 'modified');
            statusEl.title = status.inSync 
                ? 'Настройки синхронизированы с файлом'
                : `${status.differences} настроек отличаются от файла`;
        }
    } catch (error) {
        console.error('❌ Ошибка обновления статуса:', error);
        const statusEl = document.getElementById('configStatus');
        if (statusEl) {
            statusEl.className = 'config-status error';
            statusEl.title = 'Ошибка проверки статуса';
        }
    }
}

// Показ детального статуса конфигурации
async function showConfigStatus() {
    if (!configManagerReady || !window.configManager) {
        showNotification('ConfigManager не доступен', 'error');
        return;
    }
    
    try {
        const status = await window.configManager.getConfigStatus();
        const differences = status.details;
        
        let message = status.inSync 
            ? '✅ Все настройки синхронизированы с config.json'
            : `⚠️ Найдено ${status.differences} различий:\n\n`;
        
        if (!status.inSync) {
            if (differences.apikey && differences.apikey.differs) {
                message += `🔑 API Key:\n  Файл: "${differences.apikey.file}"\n  Текущий: "${differences.apikey.current}"\n\n`;
            }
            
            if (differences.settings) {
                for (let key in differences.settings) {
                    const setting = differences.settings[key];
                    if (setting.differs) {
                        message += `⚙️ ${key}: файл="${setting.file}", текущий="${setting.current}"\n`;
                    }
                }
            }
        }
        
        showNotification(message, status.inSync ? 'success' : 'warning', 5000);
    } catch (error) {
        console.error('❌ Ошибка получения статуса:', error);
        showNotification('Ошибка получения статуса', 'error');
    }
}

// Упрощенная функция load для совместимости
async function load(){
    return await refreshData();
}

// Показ/скрытие индикатора загрузки
function showLoadingIndicator(show) {
    let indicator = document.getElementById('loadingIndicator');
    
    if (show && !indicator) {
        indicator = document.createElement('div');
        indicator.id = 'loadingIndicator';
        indicator.className = 'loading-indicator';
        indicator.innerHTML = '<span>⏳ Загрузка данных...</span>';
        indicator.style.cssText = `
            position: fixed;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
            background: rgba(0, 0, 0, 0.8);
            color: white;
            padding: 12px 20px;
            border-radius: 8px;
            z-index: 10001;
            font-size: 12px;
        `;
        document.body.appendChild(indicator);
    } else if (!show && indicator) {
        indicator.remove();
    }
}

// Функция для показа уведомлений
function showNotification(message, type = 'info', duration = 3000) {
    const notification = document.createElement('div');
    notification.className = `notification ${type}`;
    notification.style.cssText = `
        position: fixed;
        top: 10px;
        right: 10px;
        padding: 10px 15px;
        border-radius: 4px;
        color: white;
        font-size: 14px;
        z-index: 10000;
        max-width: 300px;
        word-wrap: break-word;
        transition: opacity 0.3s ease;
        opacity: 0;
    `;
    
    const colors = {
        'success': '#4CAF50',
        'error': '#f44336',
        'warning': '#ff9800',
        'info': '#2196F3'
    };
    
    notification.style.backgroundColor = colors[type] || colors.info;
    
    if (message.includes('\n')) {
        notification.innerHTML = message.replace(/\n/g, '<br>');
    } else {
        notification.textContent = message;
    }
    
    document.body.appendChild(notification);
    
    setTimeout(() => notification.style.opacity = '1', 10);
    
    setTimeout(() => {
        notification.style.opacity = '0';
        setTimeout(() => {
            if (notification.parentNode) {
                document.body.removeChild(notification);
            }
        }, 300);
    }, duration);
}

// Тестирование связи с background script
async function testBackgroundConnection() {
    console.log('🔧 Проверяем подключение к background script...');
    
    try {
        const response = await sendChrome.X('get', {});
        console.log('✅ Background script отвечает:', response);
        return true;
    } catch (error) {
        console.error('❌ Background script не отвечает:', error);
        return false;
    }
}

// Основная инициализация при загрузке DOM
document.addEventListener('DOMContentLoaded', async function() {
    console.log('🚀 DOMContentLoaded - начинаем инициализацию');
    
    // Проверяем доступность Firefox API
    if (typeof browser === 'undefined') {
        console.error('❌ Firefox browser API недоступен!');
        document.body.innerHTML = `
            <div style="padding: 20px; text-align: center; color: red;">
                <h3>Ошибка загрузки</h3>
                <p>Firefox API недоступен. Пожалуйста, перезагрузите расширение.</p>
            </div>
        `;
        return;
    }
    
    createInactiveOverlay();
    applyTranslations();
    updateCSSForLanguage();
    
    showLoadingIndicator(true);
    
    try {
        // Тестируем связь с background script
        console.log('🔍 Тестируем связь с background script...');
        const connectionTest = await testBackgroundConnection();
        
        if (!connectionTest) {
            console.warn('⚠️ Background script не отвечает, пробуем продолжить...');
        }
        
        // Инициализируем ConfigManager
        await waitForConfigManager();
        
        // Загружаем данные
        if (!cachedData && !isDataLoading) {
            await preloadData();
        } else if (isDataLoading && preloadPromise) {
            await preloadPromise;
        }
        
        if (cachedData) {
            console.log('📦 Загружаем из кэша');
            await loadFromCache();
        } else {
            console.log('🌐 Кэш пуст, загружаем данные напрямую');
            await refreshData();
        }
        
        setupEventHandlers();
        
        // Обновляем статус конфигурации
        await updateConfigStatus();
        
        checkPluginState();
        
        console.log('✅ Инициализация завершена успешно');
        
        // Периодическое обновление данных
        setInterval(async () => {
            try {
                await refreshData();
                await updateConfigStatus();
            } catch (error) {
                console.warn('⚠️ Ошибка периодического обновления:', error);
            }
        }, 30000); // Каждые 30 секунд
        
    } catch (error) {
        console.error('❌ Ошибка инициализации popup:', error);
        showNotification('Ошибка загрузки данных: ' + error.message, 'error');
        
        // Показываем диагностическую информацию
        const wrapper = document.querySelector('.wrapper');
        if (wrapper) {
            const errorDiv = document.createElement('div');
            errorDiv.style.cssText = `
                padding: 10px; 
                background: #ffebee; 
                color: #c62828; 
                margin: 10px; 
                border-radius: 4px;
                font-size: 11px;
            `;
            errorDiv.innerHTML = `
                <strong>Ошибка подключения:</strong><br>
                ${error.message}<br>
                <small>Проверьте консоль для подробностей</small>
            `;
            wrapper.appendChild(errorDiv);
        }
    } finally {
        showLoadingIndicator(false);
    }
});

// Диагностическая информация при загрузке
if (typeof browser !== 'undefined') {
    console.log('🌍 Firefox расширение загружено');
    console.log('🔧 Browser API доступен:', !!browser.runtime);
    console.log('💬 SendMessage доступен:', !!browser.runtime.sendMessage);
    console.log('📦 Storage доступен:', !!browser.storage);
} else {
    console.error('❌ Firefox browser API недоступен');
}