/**
 * Cap.Guru Configuration Manager for Firefox
 * Управление настройками плагина через внешний файл конфигурации
 */

class ConfigManager {
    constructor() {
        this.configPath = 'config.json';
        this.configExists = false;
        this.defaultConfig = {
            apikey: '',
            settings: {
                onoff: '1',
                tiktok: '0',
                recap: '1', 
                oth: '0',
                binance: '0',
                funcap: '0',
                hcap: '0',
                geetest: '0',
                aws: '0'
            }
        };
    }

    /**
     * Загрузка конфигурации из файла для Firefox
     */
    async loadConfig() {
        try {
            // Проверяем доступность browser API
            if (typeof browser === 'undefined' || !browser.runtime) {
                console.warn('Browser API недоступен для загрузки конфигурации');
                this.configExists = false;
                return null;
            }

            const response = await fetch(browser.runtime.getURL(this.configPath));
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }
            
            const config = await response.json();
            this.configExists = true;
            console.log('✅ Config.json успешно загружен:', config);
            return this.validateConfig(config);
        } catch (error) {
            console.warn('Не удалось загрузить config.json:', error.message);
            console.info('Файл конфигурации отсутствует - используется ручной режим');
            this.configExists = false;
            return null;
        }
    }

    /**
     * Проверка существования файла конфигурации для Firefox
     */
    async checkConfigExists() {
        try {
            if (typeof browser === 'undefined' || !browser.runtime) {
                this.configExists = false;
                return false;
            }

            const response = await fetch(browser.runtime.getURL(this.configPath));
            this.configExists = response.ok;
            return this.configExists;
        } catch (error) {
            this.configExists = false;
            return false;
        }
    }

    /**
     * Валидация конфигурации
     */
    validateConfig(config) {
        const validatedConfig = { ...this.defaultConfig };
        
        // Проверяем API ключ
        if (config.apikey && typeof config.apikey === 'string') {
            validatedConfig.apikey = config.apikey.trim();
        }
        
        // Проверяем настройки
        if (config.settings && typeof config.settings === 'object') {
            for (let key in validatedConfig.settings) {
                if (config.settings.hasOwnProperty(key)) {
                    const value = config.settings[key];
                    // Проверяем, что значение является строкой "0" или "1"
                    if (value === '0' || value === '1' || value === 0 || value === 1) {
                        validatedConfig.settings[key] = String(value);
                    }
                }
            }
        }
        
        return validatedConfig;
    }

    /**
     * Применение конфигурации к плагину (только если файл существует)
     */
    async applyConfig(config) {
        // Если файл не существует, не применяем конфигурацию
        if (!this.configExists || !config) {
            console.log('Конфигурация не применена - файл отсутствует или ручной режим');
            return [];
        }

        const results = [];
        
        try {
            // Применяем API ключ
            if (config.apikey && config.apikey.trim() !== '') {
                const result = await this.sendToBackground('set', {
                    "n": "apikey", 
                    "v": config.apikey
                });
                results.push({ key: 'apikey', success: !!result });
            }
            
            // Применяем настройки
            if (config.settings) {
                for (let setting in config.settings) {
                    const result = await this.sendToBackground('set', {
                        "n": setting,
                        "v": config.settings[setting]
                    });
                    results.push({ key: setting, success: !!result });
                }
            }
            
            return results;
        } catch (error) {
            console.error('Ошибка при применении конфигурации:', error);
            return [];
        }
    }

    /**
     * Отправка сообщения в background script (Firefox совместимая версия)
     */
    async sendToBackground(action, data) {
        try {
            if (typeof browser === 'undefined' || !browser.runtime || !browser.runtime.sendMessage) {
                console.error('Browser runtime API недоступен');
                return null;
            }

            const response = await browser.runtime.sendMessage([action, data]);
            return response;
        } catch (error) {
            console.error('Ошибка отправки сообщения в background:', error);
            return null;
        }
    }

    /**
     * Получение текущих настроек плагина
     */
    async getCurrentSettings() {
        try {
            const response = await this.sendToBackground('get', {});
            return response && response.received ? response.received : {};
        } catch (error) {
            console.error('Ошибка при получении текущих настроек:', error);
            return {};
        }
    }

    /**
     * Сравнение конфигурации из файла с текущими настройками
     */
    async compareConfigs() {
        const fileConfig = await this.loadConfig();
        const currentSettings = await this.getCurrentSettings();
        
        // Если файл не существует, возвращаем статус "ручной режим"
        if (!this.configExists || !fileConfig) {
            return {
                manualMode: true,
                message: 'Ручной режим - файл config.json отсутствует'
            };
        }
        
        const differences = {
            manualMode: false,
            apikey: {
                file: fileConfig.apikey || '',
                current: currentSettings.apikey || '',
                differs: (fileConfig.apikey || '') !== (currentSettings.apikey || '')
            },
            settings: {}
        };
        
        for (let key in fileConfig.settings) {
            differences.settings[key] = {
                file: fileConfig.settings[key],
                current: currentSettings[key] || '0',
                differs: fileConfig.settings[key] !== (currentSettings[key] || '0')
            };
        }
        
        return differences;
    }

    /**
     * Полное обновление конфигурации
     */
    async reloadConfig() {
        console.log('🔄 Перезагрузка конфигурации...');
        
        // Сначала проверяем существование файла
        await this.checkConfigExists();
        
        if (!this.configExists) {
            console.log('📄 Файл config.json отсутствует - работаем в ручном режиме');
            return {
                config: null,
                results: [],
                success: true,
                manualMode: true,
                message: 'Ручной режим активен'
            };
        }
        
        const config = await this.loadConfig();
        const results = await this.applyConfig(config);
        
        const successCount = results.filter(r => r.success).length;
        const totalCount = results.length;
        
        console.log(`✅ Применено ${successCount}/${totalCount} настроек`);
        
        return {
            config,
            results,
            success: successCount === totalCount,
            manualMode: false
        };
    }

    /**
     * Создание резервной копии текущих настроек
     */
    async createBackup() {
        const currentSettings = await this.getCurrentSettings();
        const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
        
        const backup = {
            timestamp,
            apikey: currentSettings.apikey || '',
            settings: {}
        };
        
        const settingsKeys = ['onoff', 'tiktok', 'recap', 'oth', 'binance', 'funcap', 'hcap', 'geetest', 'aws'];
        settingsKeys.forEach(key => {
            backup.settings[key] = currentSettings[key] || '0';
        });
        
        return backup;
    }

    /**
     * Отображение состояния конфигурации
     */
    async getConfigStatus() {
        const differences = await this.compareConfigs();
        
        // Если ручной режим
        if (differences.manualMode) {
            return {
                inSync: true,
                differences: 0,
                manualMode: true,
                message: differences.message,
                details: differences
            };
        }
        
        const hasApiKeyDiff = differences.apikey.differs;
        const settingsDiffs = Object.values(differences.settings).filter(s => s.differs);
        
        return {
            inSync: !hasApiKeyDiff && settingsDiffs.length === 0,
            differences: settingsDiffs.length + (hasApiKeyDiff ? 1 : 0),
            manualMode: false,
            details: differences
        };
    }

    /**
     * Проверка режима работы
     */
    isManualMode() {
        return !this.configExists;
    }

    /**
     * Принудительное включение ручного режима
     */
    enableManualMode() {
        this.configExists = false;
        console.log('🔧 Ручной режим принудительно включен');
    }

    /**
     * Проверка совместимости с Firefox
     */
    checkFirefoxCompatibility() {
        const checks = {
            browserAPI: typeof browser !== 'undefined',
            runtime: typeof browser !== 'undefined' && !!browser.runtime,
            sendMessage: typeof browser !== 'undefined' && !!browser.runtime?.sendMessage,
            getURL: typeof browser !== 'undefined' && !!browser.runtime?.getURL
        };
        
        const compatible = Object.values(checks).every(check => check);
        
        if (!compatible) {
            console.warn('⚠️ Firefox compatibility issues detected:', checks);
        } else {
            console.log('✅ Firefox compatibility confirmed');
        }
        
        return { compatible, checks };
    }
}

// Глобальный экземпляр менеджера конфигурации
if (typeof window !== 'undefined') {
    window.configManager = new ConfigManager();
    
    // Проверяем совместимость при инициализации
    window.configManager.checkFirefoxCompatibility();
    
    console.log('✅ ConfigManager инициализирован для Firefox');
} else {
    console.warn('⚠️ Window объект недоступен, ConfigManager не инициализирован');
}